/* ***** BEGIN LICENSE BLOCK *****
 * Licensed under Version: MPL 1.1/GPL 2.0/LGPL 2.1
 * Full Terms at http://mozile.mozdev.org/0.8/LICENSE
 *
 * Software distributed under the License is distributed on an "AS IS" basis,
 * WITHOUT WARRANTY OF ANY KIND, either express or implied. See the License
 * for the specific language governing rights and limitations under the
 * License.
 *
 * The Original Code is James A. Overton's code (james@overton.ca).
 *
 * The Initial Developer of the Original Code is James A. Overton.
 * Portions created by the Initial Developer are Copyright (C) 2005-2006
 * the Initial Developer. All Rights Reserved.
 *
 * Contributor(s):
 *	James A. Overton <james@overton.ca>
 *
 * ***** END LICENSE BLOCK ***** */
 
/** Mozile Test Script
 * @fileoverview This file contains JsUnit test functions for Mozile's editing functionality.
 * <p>Project Homepage: http://mozile.mozdev.org
 * @author James A. Overton <james@overton.ca>
 * @version 0.8
 * $Id: edit.js,v 1.2 2006/08/23 16:55:28 jameso Exp $
 */

mozile.require("mozile.edit");
mozile.require("mozile.xml");
mozile.require("mozile.test.edit.util");
mozile.require("mozile.test.shared.util");
var name = "mozile.edit.*";

/**
 * Expose the functions named in this array to JsUnit.
 * Make sure to keep it up to date!
 */
function exposeTestFunctionNames() {
	//return ["testAccelerators"];
	return ["testIsEditable", "testSetStatus", "testBlock", "testIsChildless", "testIsNodeEditable", "testIsEmptyToken", "testIsEmpty", "testMark", "testAccelerators", "testConvertKeyCode", "testAddGetCommands", "testParseMES", "testFollowMESRef", "testGenerateCommand", "testGenerateCommands", "testLookupRNG", "testDone", "testUndo", "testRedo",  "testHandleEvent", "testHandleDefault", "testGetTargetAncestor", "testGetTargetDescendant", "testGetTargetNext", "testGetTargetPrevious"];
}

/**
 * 
 */
function testIsEditable() {
	var parent = document.createElement("parent");
	var element = document.createElement("element");
	var text = document.createTextNode("text");
	element.appendChild(text);
	parent.appendChild(element);

	assertFalse("The parent should not be marked as editable", mozile.edit.isEditableElement(parent));
	
	assertNull("There is no editable container", mozile.edit.getContainer(element));
	assertFalse("The element is not editable", mozile.edit.isEditable(element));
	assertFalse("The text is not editable", mozile.edit.isEditable(text));

	element.setAttribute("contentEditable", "true");
	assertEquals("The editable container is the element", element, mozile.edit.getContainer(element));
	assertFalse("The element itself is still not editable", mozile.edit.isEditable(element));
	assertTrue("The text is now editable", mozile.edit.isEditable(text));

	element.removeAttribute("contentEditable");
	element.mozile = null;
	assertNull("There is no editable container anymore", mozile.edit.getContainer(element));
	assertFalse("The element is not editable anymore", mozile.edit.isEditable(element));
	assertFalse("The text is not editable anymore", mozile.edit.isEditable(text));

	parent.setAttribute("contentEditable", "true");
	assertEquals("The editable container is the parent", parent, mozile.edit.getContainer(element));
	assertTrue("The element is inside an editable container", mozile.edit.isEditable(element));
	assertTrue("The text is editable again", mozile.edit.isEditable(text));
	
	assertTrue("The parent should be marked as editable", mozile.edit.isEditableElement(parent));
}

/**
 * 
 */
function testSetStatus() {
	return;
	assertFalse("The status should be false", mozile.edit.status);

	mozile.edit.start();
	assertTrue("The status should be true", mozile.edit.status);
	if(mozile.browser.isMozilla || mozile.browser.isOpera) {
		assertEquals("The design mode should be 'on'", "on", document.designMode);
	}
	else if(mozile.browser.isIE) {
		assertEquals("The design mode should be 'Inherit'", "Inherit", document.designMode);
	}
	else if(mozile.browser.isSafari) {
		assertEquals("The design mode should be 'off'", "off", document.designMode);
	}

	mozile.edit.stop();
	assertFalse("The status should be false", mozile.edit.status);
	if(mozile.browser.isMozilla || mozile.browser.isOpera) {
		assertEquals("The design mode should be 'off'", "off", document.designMode);
	}
	else if(mozile.browser.isIE) {
		assertEquals("The design mode should be 'Inherit'", "Inherit", document.designMode);
	}
	else if(mozile.browser.isSafari) {
		assertEquals("The design mode should be 'off'", "off", document.designMode);
	}
}

/**
 *
 */
function testBlock() {
	var target = document.getElementById("target");
	assertTrue("target is a block", mozile.edit.isBlock(target));
	var target2 = document.getElementById("target2");
	assertFalse("target2 is not a block", mozile.edit.isBlock(target2));

	assertEquals("target is target2's parentBlock", target, mozile.edit.getParentBlock(target2));

}

/**
 * This test relies on the schema.
 */
function testIsChildless() {
	var target = document.getElementById("target");
	assertFalse("target is not childless", mozile.edit.isChildless(target));
	
	var br = document.createElement("br");
	assertTrue("br is childless", mozile.edit.isChildless(br));
}

/**
 * This test relies on the schema.
 */
function testIsNodeEditable() {
	var target = document.getElementById("target");
	assertTrue("target is editable", mozile.edit.isNodeEditable(target));
	assertTrue("target.firstChild is editable", mozile.edit.isNodeEditable(target.firstChild));
	
	var br = document.createElement("br");
	assertFalse("br is editable", mozile.edit.isNodeEditable(br));
}

/**
 *
 */
function testIsEmptyToken() {
	assertTrue("This is an empty token", mozile.edit.isEmptyToken(document.createTextNode(mozile.emptyToken)));
	assertFalse("This is not an empty token", mozile.edit.isEmptyToken(document.createTextNode(mozile.emptyToken +"text")));
}

/**
 *
 */
function testIsEmpty() {
	assertFalse("This node has non-white-space text", mozile.edit.isEmpty(document.createTextNode("text")));
	assertFalse("This node is an empty token", mozile.edit.isEmpty(document.createTextNode(mozile.emptyToken)));

	assertTrue("This node has no text", mozile.edit.isEmpty(document.createTextNode("")));
	assertTrue("This node has only white-space text", mozile.edit.isEmpty(document.createTextNode(" 	")));

	var element = document.createElement("element");
	assertTrue("This element is empty", mozile.edit.isEmpty(element));
	var text = document.createTextNode("");
	element.appendChild(text);
	assertTrue("This element has only empty children", mozile.edit.isEmpty(element));
	text.appendData("text");
	assertFalse("This element now has text", mozile.edit.isEmpty(element));
}

/**
 * Test setting and getting marks for an element.
 */
function testMark() {
	var target = document.getElementById("target");
	
	assertTrue("There should be no 'mozile' property", target.mozile == undefined);
	var result = mozile.edit.setMark(target, "test", "foo");

	assertEquals("The result should be 'foo'", "foo", result);
	assertEquals("There should now be a 'mozile' object", "object", typeof(target.mozile));
	assertEquals("The value of 'test' should be 'foo'", "foo", target.mozile["test"]);

	assertEquals("getMark should return 'foo'", "foo", mozile.edit.getMark(target, "test"));

	var bar = document.documentElement;
	result = mozile.edit.setMark(target, "test2", bar);
	assertEquals("The result should be bar", bar, result);
	assertEquals("The value of 'test2' should be bar", bar, target.mozile["test2"]);
	assertEquals("getMark should return bar", bar, mozile.edit.getMark(target, "test2"));
	
	// TODO: Test for and XML target element. Will fail under IE7.
}

/**
 * 
 */
function testAccelerators() {
	var accel = "Command-Shift-H";
	var charCode = 72; // 'H'
	
	// Test splitAccelerators
	var accels = ["Left", "Right", "Command-Shift-H"];
	var newAccels = mozile.edit.splitAccelerators("  Left   Right	Command-Shift-H  ");
	for(var i=0; i < accels.length; i++) {
		assertEquals("Accel "+ i +" should match.", accels[i], newAccels[i]);
	}
	
	// Test parseAccelerator
	var obj = mozile.edit.parseAccelerator(accel);
	assertEquals("'command' should be true",  true,  obj.command);
	assertEquals("'meta' should be false",    false, obj.meta);
	assertEquals("'ctrl' should be false",    false, obj.ctrl);
	assertEquals("'alt' should be false",     false, obj.alt);
	assertEquals("'shift' should be true",    true,  obj.shift);
	assertEquals("'character' should be 'H'", "H",   obj.character);
	
	// Test generateAccelerator
	var event = {
		type: "keypress", 
		keyCode: charCode,
		charCode: charCode,
		metaKey: false,
		ctrlKey: false, 
		altKey: false,
		shiftKey: true
	}
	if(mozile.browser.isIE) event.type = "keydown";
	if(mozile.os.isMac) event.metaKey = true;
	else event.ctrlKey = true;
	var newAccel = mozile.edit.generateAccelerator(event);
	assertEquals("The newAccel should match", "Command-Shift-H", newAccel);
	
	// checkAccelerators is trivial
	
	// Test checkAccelerator
	event.shiftKey = false;
	assertFalse("The accelerator is not active: no shift key", mozile.edit.checkAccelerator(event, accel));

	event.accel = null;
	event.shiftKey = true;
	assertTrue("The accelerator is active", mozile.edit.checkAccelerator(event, accel));

	event.accel = null;
	event.altKey = true;
	assertFalse("The accelerator is not active: alt key", mozile.edit.checkAccelerator(event, accel));
	event.altKey = false;

	event.accel = null;
	event.charCode = 99;
	assertFalse("The accelerator is not active: wrong character", mozile.edit.checkAccelerator(event, accel));
	event.charCode = charCode;
}

/**
 * 
 */
function testConvertKeyCode() {
	// Currently trivial.
}

/**
 * 
 */
function testAddGetCommands() {
	var name = "testCommand";	
	assertNull("No command '"+ name +"' yet.", mozile.edit.getCommand(name));
	
	// Automatically adds the command.
	var command = new mozile.edit.Command(name);
	assertEquals("Now there should be a command", command, mozile.edit.getCommand(name));
	delete mozile.edit.allCommands[name];
}

/**
 * 
 */
function testParseMES() {
	// Create a partial MES
	var doc = mozile.xml.parse('<mes xmlns="http://mozile.mozdev.org/ns/mes/1.0"/>');
	var MES = doc.documentElement;
	var command1 = mozile.dom.createElementNS(mozile.xml.ns.mes, "command");
	command1.setAttribute("name", "command1");
	MES.appendChild(command1);
	var group2 = mozile.dom.createElementNS(mozile.xml.ns.mes, "group");
	group2.setAttribute("name", "group2");
	MES.appendChild(group2);
	var command2 = mozile.dom.createElementNS(mozile.xml.ns.mes, "command");
	command2.setAttribute("name", "command2");
	group2.appendChild(command2);
	
	// Parse the MES
	var group1 = new mozile.edit.CommandGroup("group1");
	mozile.edit.parseMES(group1, MES);
	
	assertEquals("There should be two commands", 2, group1._commands.length);
	var c1 = group1._commands[0];
	var c2 = group1._commands[1];
	assertEquals("c1 should be named command1", "command1", c1.name);
	assertEquals("c2 should be named group2", "group2", c2.name);
	assertEquals("The group should have one command", 1, c2._commands.length);
	var c3 = c2._commands[0];
	assertEquals("c3 should be named command2", "command2", c3.name);
}

/**
 * 
 */
function testFollowMESRef() {
	// Create a partial MES
	var doc = mozile.xml.parse('<mes xmlns="http://mozile.mozdev.org/ns/mes/1.0"/>');
	var MES = doc.documentElement;
	var define = mozile.dom.createElementNS(mozile.xml.ns.mes, "define");
	define.setAttribute("name", "target");
	MES.appendChild(define);
	var ref = mozile.dom.createElementNS(mozile.xml.ns.mes, "ref");
	ref.setAttribute("name", "target");
	MES.appendChild(ref);
	var ref2 = mozile.dom.createElementNS(mozile.xml.ns.mes, "ref");
	ref2.setAttribute("name", "none");
	MES.appendChild(ref2);
	
	// Follow refs
	var result = mozile.edit.followMESRef(ref);
	assertEquals("The define element should be returned", MES.firstChild, result);

	result = mozile.edit.followMESRef(ref2);
	assertNull("No element should be returned", result);
}

/**
 * 
 */
function testGenerateCommand() {
	// Command case.
	var nodeString = '<command xmlns="http://mozile.mozdev.org/ns/mes/1.0" name="testCommand" class="Command" accel="Command-U" label="Underline" image="silk/text_underline" tooltip="Underline text"><element><span/></element><script><![CDATA[this.test = true;]]></script></command>';
	var node = mozile.xml.parseElement(nodeString);
	var command = mozile.edit.generateCommand(node);
	
	assertEquals("name should be 'testCommand'", "testCommand", command.name);
	assertEquals("constructor should be mozile.edit.Command", mozile.edit.Command, command.constructor);
	assertEquals("accel should be 'Command-U'", "Command-U", command.accel);
	assertEquals("label should be 'Underline'", "Underline", command.label);
	assertEquals("image should be 'silk/text_underline'", "silk/text_underline", command.image);
	assertEquals("tooltip should be 'Underline text'", "Underline text", command.tooltip);

	assertNotUndefined("The element should be defined", command.element);
	assertEquals("element nodeName should be 'span'", "span", command.element.nodeName);

	assertEquals("The 'test' property should have been defined by the script", true, command.test);

	delete mozile.edit.allCommands["testCommand"];


	// Group case.
	nodeString = '<group xmlns="http://mozile.mozdev.org/ns/mes/1.0" name="testCommand" label="Fonts" image="silk/font" tooltip="Select a font"/>';
	node = mozile.xml.parseElement(nodeString);
	command = mozile.edit.generateCommand(node);
	
	assertEquals("name should be 'testCommand'", "testCommand", command.name);
	assertEquals("constructor should be mozile.edit.CommandGroup", mozile.edit.CommandGroup, command.constructor);
	assertEquals("label should be 'Fonts'", "Fonts", command.label);
	assertEquals("image should be 'silk/font'", "silk/font", command.image);
	assertEquals("tooltip should be 'Select a font'", "Select a font", command.tooltip);

	delete mozile.edit.allCommands["testCommand"];

}

/**
 * 
 */
function testGenerateCommands() {
	// Currently trivial.
}

/**
 * 
 */
function testLookupRNG() {
	// Currently trivial.
}

/**
 * 
 */
function testDone() {
	// Currently trivial.
}

/**
 * 
 */
function testUndo() {
	// Currently trivial.
}

/**
 * 
 */
function testRedo() {
	// Currently trivial.
}

/**
 * 
 */
function testHandleEvent() {
	// Currently trivial.
}

/**
 * 
 */
function testHandleDefault() {
	// Currently trivial.
}

/**
 * Search for targets in the default "ancestor" direction.
 */
function testGetTargetAncestor() {
	var command = {};
	var target, node;
	var container = document.getElementById("container");
	var target1 = document.getElementById("target");
	var target2 = document.getElementById("target2");
	var text = target2.firstChild
	mozile.test.shared.preselect();
	var selection = mozile.dom.selection.get();
	selection.collapse(text, 5);

	target = "any";
	node = mozile.edit._getTarget(null, target);
	assertEquals(target +" should return text", text, node);

	target = "text";
	node = mozile.edit._getTarget(null, target);
	assertEquals(target +" should return text (2)", text, node);

	target = "element";
	node = mozile.edit._getTarget(null, target);
	assertEquals(target +" should return target2", target2, node);

	target = "block";
	node = mozile.edit._getTarget(null, target);
	assertEquals(target +" should return target1", target1, node);

	target = "localName span";
	node = mozile.edit._getTarget(null, target);
	assertEquals(target +" should return target2 (2)", target2, node);

	target = "localName p";
	node = mozile.edit._getTarget(null, target);
	assertEquals(target +" should return target1 (2)", target1, node);
}

/**
 * Search for targets in the "descendant" direction.
 */
function testGetTargetDescendant() {
	var command = {};
	var direction, target, node;
	var container = document.getElementById("container");
	var target1 = document.getElementById("target");
	var target2 = document.getElementById("target2");
	var target3 = document.getElementById("target3");
	var text = target1.firstChild
	mozile.test.shared.preselect();
	var selection = mozile.dom.selection.get();
	var event = {node: target1};

	direction = "descendant";
	target = "any";
	node = mozile.edit._getTarget(event, target, direction);
	assertEquals(target +" should return text", text, node);

	target = "text";
	node = mozile.edit._getTarget(event, target, direction);
	assertEquals(target +" should return text", text, node);

	target = "element";
	node = mozile.edit._getTarget(event, target, direction);
	assertEquals(target +" should return target2 (2)", target2, node);

	target = "block";
	node = mozile.edit._getTarget(event, target, direction);
	assertEquals(target +" should return null", null, node);

	target = "localName span";
	node = mozile.edit._getTarget(event, target, direction);
	assertEquals(target +" should return target2 (3)", target2, node);

	target = "localName code";
	node = mozile.edit._getTarget(event, target, direction);
	assertEquals(target +" should return target3", target3, node);
}

/**
 * Search for targets in the "next" direction.
 */
function testGetTargetNext() {
	var command = {};
	var direction, target, node;
	var container = document.getElementById("container");
	var target1 = document.getElementById("target");
	var target2 = document.getElementById("target2");
	var p1 = document.getElementById("p1");
	var firstItem = document.getElementById("firstItem");
	var text = target1.firstChild
	mozile.test.shared.preselect();
	var selection = mozile.dom.selection.get();
	selection.collapse(text, 5);

	direction = "next";
	target = "any";
	node = mozile.edit._getTarget(null, target, direction);
	assertEquals(target +" should return target2", target2, node);

	target = "text";
	node = mozile.edit._getTarget(null, target, direction);
	assertEquals(target +" should return target2.firstChild", target2.firstChild, node);

	target = "element";
	node = mozile.edit._getTarget(null, target, direction);
	assertEquals(target +" should return target2 (2)", target2, node);

	target = "block";
	node = mozile.edit._getTarget(null, target, direction);
	assertEquals(target +" should return firstItem", firstItem, node);

	target = "localName span";
	node = mozile.edit._getTarget(null, target, direction);
	assertEquals(target +" should return target2 (3)", target2, node);

	target = "localName p";
	node = mozile.edit._getTarget(null, target, direction);
	assertEquals(target +" should return p1", p1, node);
}

/**
 * Search for targets in the "previous" direction.
 */
function testGetTargetPrevious() {
	var command = {};
	var direction, target, node;
	var container = document.getElementById("container");
	var target1 = document.getElementById("target");
	var target2 = document.getElementById("target2");
	var previousBlock = mozile.dom.getFirstChildElement(container);
	var text = target1.childNodes[2];
	mozile.test.shared.preselect();
	var selection = mozile.dom.selection.get();
	selection.collapse(text, 5);

	direction = "previous";
	target = "any";
	node = mozile.edit._getTarget(null, target, direction);
	assertEquals(target +" should return  target2.firstChild",  target2.firstChild, node);

	target = "text";
	node = mozile.edit._getTarget(null, target, direction);
	assertEquals(target +" should return target2.firstChild (2)", target2.firstChild, node);

	target = "element";
	node = mozile.edit._getTarget(null, target, direction);
	assertEquals(target +" should return target2", target2, node);

	target = "block";
	node = mozile.edit._getTarget(null, target, direction);
	assertEquals(target +" should return previousBlock", previousBlock, node);

	target = "localName span";
	node = mozile.edit._getTarget(null, target, direction);
	assertEquals(target +" should return target2 (2)", target2, node);

	target = "localName p";
	node = mozile.edit._getTarget(null, target, direction);
	assertEquals(target +" should return previousBlock (2)", previousBlock, node);
}




