/* ***** BEGIN LICENSE BLOCK *****
 * Licensed under Version: MPL 1.1/GPL 2.0/LGPL 2.1
 * Full Terms at http://mozile.mozdev.org/0.8/LICENSE
 *
 * Software distributed under the License is distributed on an "AS IS" basis,
 * WITHOUT WARRANTY OF ANY KIND, either express or implied. See the License
 * for the specific language governing rights and limitations under the
 * License.
 *
 * The Original Code is James A. Overton's code (james@overton.ca).
 *
 * The Initial Developer of the Original Code is James A. Overton.
 * Portions created by the Initial Developer are Copyright (C) 2005-2006
 * the Initial Developer. All Rights Reserved.
 *
 * Contributor(s):
 *	James A. Overton <james@overton.ca>
 *
 * ***** END LICENSE BLOCK ***** */
 
/** Mozile Test Script
 * @fileoverview This file contains JsUnit test functions for Mozile's Remove Text command.
 * <p>Project Homepage: http://mozile.mozdev.org
 * @author James A. Overton <james@overton.ca>
 * @version 0.8
 * $Id: Insert.js,v 1.2 2006/08/23 16:55:28 jameso Exp $
 */

mozile.require("mozile.edit.rich");
mozile.require("mozile.test.shared.util");
var name = "mozile.edit.Insert";

/**
 * Expose the functions named in this array to JsUnit.
 * Make sure to keep it up to date!
 */
function exposeTestFunctionNames() {
	//return ["testText"];
	return ["testText", "testCollapsed", "testCollapsedRemove", "testRange", "testRangeRemove"];
}


/**
 * Insert some text.
 */
function testText() {
	mozile.test.shared.preselect();
	var name = "testCommand";
	var command = new mozile.edit.Insert(name);
	command.remove = false;

	assertEquals("Name should be '"+ name +"'", name, command.name);
	assertEquals("Group should be false", false, command.group);

	var target = document.getElementById("target");
	var text = target.firstChild;
	var original = text.data;

	var selection = mozile.dom.selection.get();
	selection.collapse(text, 5);
	
	var event = { };
	
	// Execute
	var i = 1;
	var insert = "sampleText";
	command.text = insert;
	var state = command.prepare(event);
	assertNotUndefined(i +" There should be a state");
	assertNull(i +" There should not be a state.element", state.element);
	assertNotUndefined(i +" There should be a state.text", state.text);
	state = command.execute(state);
	assertTrue(i +" The command should have been executed.", state.executed);
	assertEquals(i +" Check the new text", original.substring(0,5) + insert + original.substring(5), text.data);
	
	// Unexecute
	state = command.unexecute(state);
	assertFalse("The command should have been unexecuted.", state.executed);
	assertEquals("Check the text node contents after unexecution", original, text.data);
	
	// Execute
	i = 2;
	insert = "&#x000A;";
	command.text = insert;
	state = command.prepare(event);
	assertNotUndefined(i +" There should be a state");
	assertNull(i +" There should not be a state.element", state.element);
	assertNotUndefined(i +" There should be a state.text", state.text);
	state = command.execute(state);
	assertTrue(i +" The command should have been executed.", state.executed);
	assertEquals(i +" Check the new text", original.substring(0,5) + insert + original.substring(5), text.data);
	
	// Unexecute
	state = command.unexecute(state);
	assertFalse("The command should have been unexecuted.", state.executed);
	assertEquals("Check the text node contents after unexecution", original, text.data);
	
	delete mozile.edit.allCommands[name];
}


/**
 * Insert a new node with an empty text node, and move the selection in to it.
 */
function testCollapsed() {
	mozile.test.shared.preselect();
	var name = "testCommand";
	var command = new mozile.edit.Insert(name);
	var elementName = "element";
	command.element = mozile.dom.createElement(elementName);
	command.remove = false;

	assertEquals("Name should be '"+ name +"'", name, command.name);
	assertEquals("Group should be false", false, command.group);

	var target = document.getElementById("target");
	var text = target.firstChild;
	var original = text.data;

	var selection = mozile.dom.selection.get();
	selection.collapse(text, 5);
	
	var event = { };
	
	// Execute
	var state = command.prepare(event);
	assertNotUndefined("There should be a state");
	assertNotUndefined("There should be a state.element", state.element);
	assertEquals("The state.element should have the right name", elementName, mozile.dom.getLocalName(state.element).toLowerCase());
	
	state = command.execute(state);
	assertTrue("The command should have been executed.", state.executed);		
	assertEquals("Check the first text node contents after execution", 
		original.substring(0, 5), target.firstChild.data);
	assertEquals("The element should have been inserted", state.element, target.childNodes[1]);
	assertEquals("The element should have an empty text node", "", target.childNodes[1].firstChild.data);
	assertEquals("Check the second text node contents after execution", 
		original.substring(5), target.childNodes[2].data);
	
	assertEquals("The selection should be in the new element", state.element.lastChild, selection.focusNode);
	assertEquals("The selection offset should be 0", 0, selection.focusOffset);
	assertEquals("The selection should be collapsed", true, selection.isCollapsed);
	
	// Unexecute
	state = command.unexecute(state);
	assertFalse("The command should have been unexecuted.", state.executed);
	assertEquals("Check the text node contents after unexecution", 
		original, text.data);

	if(state.element.parentNode) assertEquals("If the state.element has a parent, it has to be a document fragment", mozile.dom.DOCUMENT_FRAGMENT_NODE, state.element.parentNode.nodeType);
	else assertEquals("Check that the state.element has been removed", null, state.element.parentNode);
	
	delete mozile.edit.allCommands[name];
}

/**
 * Insert a new node, and move the selection to the right.
 */
function testCollapsedRemove() {
	mozile.test.shared.preselect();
	var name = "testCommand";
	var command = new mozile.edit.Insert(name);
	var elementName = "element";
	command.element = mozile.dom.createElement(elementName);
	command.remove = true;

	assertEquals("Name should be '"+ name +"'", name, command.name);
	assertEquals("Group should be false", false, command.group);

	var target = document.getElementById("target");
	var text = target.firstChild;
	var original = text.data;

	var selection = mozile.dom.selection.get();
	selection.collapse(text, 5);
	
	var event = { };
	
	// Execute
	var state = command.prepare(event);
	assertNotUndefined("There should be a state");
	assertNotUndefined("There should be a state.element", state.element);
	assertEquals("The state.element should have the right name", elementName, mozile.dom.getLocalName(state.element).toLowerCase());
	
	state = command.execute(state);
	assertTrue("The command should have been executed.", state.executed);		
	assertEquals("Check the first text node contents after execution", 
		original.substring(0, 5), target.firstChild.data);
	assertEquals("The element should have been inserted", state.element, target.childNodes[1]);
	assertEquals("The element should be empty", 0, target.childNodes[1].childNodes.length);
	assertEquals("Check the second text node contents after execution", 
		original.substring(5), target.childNodes[2].data);
	
	assertEquals("The selection should be in the next text element", target.childNodes[2], selection.focusNode);
	assertEquals("The selection offset should be 0", 0, selection.focusOffset);
	assertEquals("The selection should be collapsed", true, selection.isCollapsed);
	
	// Unexecute
	state = command.unexecute(state);
	assertFalse("The command should have been unexecuted.", state.executed);
	assertEquals("Check the text node contents after unexecution", 
		original, text.data);

	if(state.element.parentNode) assertEquals("If the state.element has a parent, it has to be a document fragment", mozile.dom.DOCUMENT_FRAGMENT_NODE, state.element.parentNode.nodeType);
	else assertEquals("Check that the state.element has been removed", null, state.element.parentNode);
	
	delete mozile.edit.allCommands[name];
}


/**
 * Move the contents of the range into a new node.
 */
function testRange() {
	mozile.test.shared.preselect();
	var name = "testCommand";
	var command = new mozile.edit.Insert(name);
	var elementName = "element";
	command.element = mozile.dom.createElement(elementName);
	command.remove = false;

	var target = document.getElementById("target");
	var target2 = document.getElementById("target2");
	var original = target.firstChild.data;

	var selection = mozile.dom.selection.get();
	selection.collapse(target.firstChild, 5);
	selection.extend(target2.firstChild, 5);
	
	var event = { };
	
	// Execute
	var state = command.prepare(event);
	assertNotUndefined("There should be a state");
	assertNotUndefined("There should be a state.element", state.element);
	assertEquals("The state.element should have the right name", elementName, mozile.dom.getLocalName(state.element).toLowerCase());
	
	state = command.execute(state);
	assertTrue("The command should have been executed.", state.executed);		
	assertEquals("Check the text node contents after execution", 
		original.substring(0, 5), target.firstChild.data);
	assertEquals("The element should have been inserted", state.element, target.childNodes[1]);
	assertEquals("The third child should be like target2", target2.nodeName, target.childNodes[2].nodeName);
	
	assertEquals("Execute: The selection anchor should be in the new element", state.element.firstChild, selection.anchorNode);
	assertEquals("Execute: The selection offset should be 0", 0, selection.anchorOffset);
	assertEquals("Execute: The selection should be in the new element", state.element.lastChild.firstChild, selection.focusNode);
	assertEquals("Execute: The selection offset should be 5", 5, selection.focusOffset);
	assertFalse("Execute: The selection should not be collapsed", selection.isCollapsed);
	
	// Unexecute
	state = command.unexecute(state);
	assertFalse("The command should have been unexecuted.", state.executed);
	assertEquals("Check the text node contents after unexecution", 
		original, target.firstChild.data);
	
	assertEquals("Unexecute: The selection anchor should be in target's firstChild", target.firstChild, selection.anchorNode);
	assertEquals("Unexecute: The selection offset should be 5", 5, selection.anchorOffset);
	assertEquals("Unexecute: The selection should be in target2's firstChild", target2.firstChild, selection.focusNode);
	assertEquals("Unexecute: The selection offset should be 5", 5, selection.focusOffset);
	assertFalse("Unexecute: The selection should not be collapsed", selection.isCollapsed);

	if(state.element.parentNode) assertEquals("If the state.element has a parent, it has to be a document fragment", mozile.dom.DOCUMENT_FRAGMENT_NODE, state.element.parentNode.nodeType);
	else assertEquals("Check that the state.element has been removed", null, state.element.parentNode);
	
	delete mozile.edit.allCommands[name];
}



/**
 * Remove the contents of a range before inserting.
 */
function testRangeRemove() {
	var name = "testCommand";
	var command = new mozile.edit.Insert(name);
	var elementName = "element";
	command.element = mozile.dom.createElement(elementName);
	command.remove = true;
	command.prompt = function(event, state) { 
		state.test = true; 
		return true;
	}

	assertEquals("Name should be '"+ name +"'", name, command.name);
	assertEquals("Group should be false", false, command.group);

	mozile.test.shared.preselect();
	var target = document.getElementById("target");
	var original = target.firstChild.data;

	var selection = mozile.dom.selection.get();
	selection.collapse(target.firstChild, 5);
	selection.extend(target.firstChild, 15);
	
	var event = { };
	
	// Execute
	var state = command.prepare(event);
	assertNotUndefined("There should be a state")
	assertNotUndefined("There should be a state.element", state.element);
	assertEquals("The state.element should have the right name", elementName, mozile.dom.getLocalName(state.element).toLowerCase());
	assertEquals("test should be true", true, state.test);
	
	state = command.execute(state);
	assertTrue("The command should have been executed.", state.executed);		
	assertEquals("Check the text node contents after execution", 
		original.substring(0, 5), target.firstChild.data);
	assertEquals("The element should hve been inserted", state.element, target.childNodes[1]);
	assertEquals("Check the text node contents after execution", 
		original.substring(15), target.childNodes[2].data);

	assertEquals("Execute: The selection should be in the next sibling", state.element.nextSibling, selection.focusNode);
	assertEquals("Execute: The selection offset should be 0", 0, selection.focusOffset);
	assertEquals("Execute: The selection should be collapsed", true, selection.isCollapsed);
	
	// Unexecute
	state = command.unexecute(state);
	assertFalse("The command should have been unexecuted.", state.executed);
	assertEquals("Check the text node contents after unexecution", 
		original, target.firstChild.data);

	if(state.element.parentNode) assertEquals("If the state.element has a parent, it has to be a document fragment", mozile.dom.DOCUMENT_FRAGMENT_NODE, state.element.parentNode.nodeType);
	else assertEquals("Check that the state.element has been removed", null, state.element.parentNode);
	
	assertEquals("Unexecute: The selection anchor should be in target's firstChild", target.firstChild, selection.anchorNode);
	assertEquals("Unexecute: The selection offset should be 5", 5, selection.anchorOffset);
	assertEquals("Unexecute: The selection should be in target's firstChild", target.firstChild, selection.focusNode);
	assertEquals("Unexecute: The selection offset should be 15", 15, selection.focusOffset);
	assertFalse("Unexecute: The selection should not be collapsed", selection.isCollapsed);

	delete mozile.edit.allCommands[name];
}

