/** Mozile Component Test Script
 * @fileoverview This file contains JSUnit test functions for testing the MozileMediator object.
 * Requires MozileComponentUtils.js
 * 
 * @link http://mozile.mozdev.org 
 * @author James A. Overton <james@overton.ca>
 * @version 0.7.0
 */


var name = "MozileMediator";
var root = "/this/root/";
var warnBeforeUnload = false;
var configString = name+": root='"+root+"', warnBeforeUnload="+warnBeforeUnload;

var mozillaVersion = navigator.userAgent.match(/rv\:(\d+\.\d+)/)[1];

/** Test - Constructor
 * Test the constructor and basic methods.
 */
function testConstructor() {
	var component = mozile;
	checkComponentBasics(component);
	checkMediatorBasics(component);
	checkInterfaces(component);
}


/** Test - Options
 * Test getting and setting options.
 */
function testOptions() {
	var component = mozile;
	checkOptions(component);
}


/**
 * Test the list functions: getItems, getItem, addItem
 */
function testLists() {
	var component = mozile;

	// Older versions of Mozila choke on this test, although they seem to work fine.
	if(Number(mozillaVersion) < 1.8) return;
	
	var methods = [
		["Resource", new MozileResource("script", "test"), "test"], 
		["Module", new MozileModule("Test"), "Test"], 
		["SaveModule", new MozileModule("SaveTest"), "SaveTest"], 
		["Command", new MozileCommand("Test: id=test"), "test"], 
	];
	var method, object, name;
	for(var m=0; m < methods.length; m++) {
		method = methods[m][0];
		object = methods[m][1];
		name = methods[m][2];
		assertNotNull("get"+method+"s() should not be null", component["get"+method+"s"]());
		assertNotUndefined("get"+method+"s() should be defined.", component["get"+method+"s"]());

		assertEquals("get"+method+"s() should return an object.", "[object Object]", component["get"+method+"s"]().toString());
		assertUndefined("No object should be returned by get"+method+"('"+name+"')", component["get"+method](name));
		assertEquals("The added object should be returned by add"+method+"()", object, component["add"+method](object));
		assertEquals("The object should be returned by get"+method+"('"+name+"')", object, component["get"+method](name));
		
	}

		// Accelerators
	assertNotNull("getAccelerators() should not be null", component.getAccelerators());
	assertNotUndefined("getAccelerators() should be defined.", component.getAccelerators());
	assertEquals("getAccelerators() should return an object.", "[object Object]", component.getAccelerators().toString());
	
	object = new MozileCommand("Test: id=test2, accel=Command-T");		
	assertUndefined("No object should be returned by getAccelerator('Command-T')", component.getAccelerator("Command-T"));
	assertEquals("The added object should be returned by add"+method+"()", object, component.addAccelerator("Command-T",object));
	assertEquals("The object should be returned by getAccelerator('Command-T')", object, component.getAccelerator("Command-T"));
	
	//	Test getEditors
	assertNotNull("getEditors() should not be null", component.getEditors());
	assertNotUndefined("getEditors() should be defined.", component.getEditors());
	
	var element = document.getElementById("target2");
	assertEquals("The getEditors array should have length 1", 1, component.getEditors().length);
	assertEquals("The element should be an editor", false, component.isEditor(element));
	assertEquals("addEditor should return the element", element, component.addEditor(element));
	assertEquals("The getEditors array should have length 2", 2, component.getEditors().length);
	assertEquals("The element should be an editor", true, component.isEditor(element));
	
	assertUndefined("The currentEditor should be undefined", component.getCurrentEditor());
	assertEquals("setCurrentEditor should return the element", element, component.setCurrentEditor(element));
	assertEquals("getCurrentEditor should now return the element", element, component.getCurrentEditor(element));	
	
}


/** 
 * Test the cleanup functions.
 */
function testCleanUp() {
	var component = mozile;
	
	var clone = document.documentElement.cloneNode(true);	

	var count1 = clone.getElementsByTagName("script").length;
	clone = component._cleanDOM(clone);
	var count2 = clone.getElementsByTagName("script").length;
	
	assertEquals("There should be two fewer script elements", count1-3, count2);
	
	var serializer = new XMLSerializer;
	var content = serializer.serializeToString(clone);
	
	assertEquals("There should be no match for 'Mozile-Core-core.js'", -1, content.indexOf("Mozile-Core-core.js"));


// _cleanInterfaces

}




/** Test - Check Mediator Basics
 * Check basic getting methods.
 */
function checkMediatorBasics(component) {
	assertEquals("root should be '../../'", "../../", component.getRoot());
	assertEquals("mode should be 'XHTML'", "XHTML", component.getMode());
	assertEquals("debugLevel should be 5", 5, component.getDebugLevel());
	
	assertNotNull("getStyleSheet() should not be null", component.getOptions());
	assertNotUndefined("getStyleSheet() should be defined.", component.getOptions());
	// Older versions of Mozila choke on this test, although they seem to work fine.
	if(Number(mozillaVersion) >= 1.8) assertEquals("getStyleSheet() should be a CSSStyleObject.", "[object CSSStyleSheet]", component.getStyleSheet().toString());
	
	assertNotNull("getCommandList() should not be null", component.getCommandList());
	assertNotUndefined("getCommandList() should be defined.", component.getCommandList());
	// Older versions of Mozila choke on this test, although they seem to work fine.
	if(Number(mozillaVersion) >= 1.8) assertEquals("getCommandList() should be a MozileCommandList.", "[object MozileCommandList]", component.getCommandList().toString());
	
	assertEquals("getOperatingSystem should be a string", typeof(""), typeof(component.getOperatingSystem()));
	assertEquals("getBrowserName should be a string", typeof(""), typeof(component.getBrowserName()));
	assertEquals("getBrowserVersion should be a string", typeof(""), typeof(component.getBrowserVersion()));
	assertEquals("getGeckoVersion should be a string", typeof(""), typeof(component.getGeckoVersion()));
	assertEquals("getMozillaVersion should be a string", typeof(""), typeof(component.getMozillaVersion()));
	
	assertEquals("This is not an extension.", false, component.isExtension());
	assertEquals("This page is not enhanced.", false, component.isEnhanced());
	
		// Strange fix for statusbar
	component.getStatusbar()._elementOrPath = "../core/status.xml";
	
	assertEquals("Not editable by default.", false, component.isEditable());
	component.startEditing();
	assertEquals("Editing should have started.", true, component.isEditable());
	component.stopEditing();
	assertEquals("Editing should hae stopped.", false, component.isEditable());
}


/**
 * Check the various interfaces Mozile defines.
 */
function checkInterfaces(component) {
	var interfaces = ["AboutInterface", "MessageInterface", "SaveInterface", "SourceInterface", "Toolbar", "Statusbar"];

		// Strange fix for statusbar
	component.getStatusbar()._elementOrPath = "../core/status.xml";
	
	for(var i=0; i < interfaces.length; i++) {
		assertEquals("The "+ interfaces[i] + " should be an object.", "object", typeof(eval("component.get"+interfaces[i]+"()")));
		assertEquals("The "+ interfaces[i] + ".getElement() should be an object.", "object", typeof(eval("component.get"+interfaces[i]+"().getElement()")));
	}
}


