/** Mozile Link Resource Test Link
 * @fileoverview This file contains JSUnit test functions for testing the MozileLinkResource class. 
 * Requires MozileResourcesUtils.js
 * 
 * @link http://mozile.mozdev.org 
 * @author James A. Overton <james@overton.ca>
 * @version 0.7.2
 */



var category = "link";
var id = "testLink";
var source = "http://source.com/";
var defaultType = "text/css";
var type = "text/CSS";
var defaultRelation = "stylesheet";
var relation = "relation";
var defaultNamespace = XHTMLNS;
var namespace = "http://namespace.com/";
var content = "Test content.";


/** Test - Constructor
 * Make sure that the MozileLinkResource constructor is working. Use the default namespace.
 */
function testConstructor() {
	var resource = new MozileLinkResource(id, source);
	checkResource(resource);
	checkLinkResource(resource);
}


/** Test - Constructor
 * Make sure that the MozileLinkResource constructor is working. Define a type and a namespace.
 */
function testConstructorNS() {
	var resource = new MozileLinkResource(id, source, relation, type, namespace);
	checkResourceNS(resource);
	checkLinkResourceNS(resource);
}



/** Test - Create Element
 * Create an element using the default namespace.
 */
function testCreateElement() {
	var resource = new MozileLinkResource(id, source);

		// getElement creates a new element	
	var element1 = resource.getElement();
	checkElement(element1);
	checkLinkElement(element1);

		// createElement creates a new element to replace the old one
	var element2 = resource.createElement();
	checkElement(element2);
	checkLinkElement(element2);

		// getElement should get element2
	var element3 = resource.getElement();
	checkElement(element3);
	checkLinkElement(element3);
	
	compareElements(element1, element2, element3);
}


/** Test - Create Element
 * Create an element, specifying a type and a namespace.
 */
function testCreateElementNS() {
	var resource = new MozileLinkResource(id, source, relation, type, namespace);

		// getElement creates a new element	
	var element1 = resource.getElement();
	checkElementNS(element1);
	checkLinkElementNS(element1);

		// createElement creates a new element to replace the old one
	var element2 = resource.createElement();
	checkElementNS(element2);
	checkLinkElementNS(element2);

		// getElement should get element2
	var element3 = resource.getElement();
	checkElementNS(element3);
	checkLinkElementNS(element3);
	
	compareElements(element1, element2, element3);
}


/** Test - Content
 * Test the content methods.
 */
function testContent() {
	var resource = new MozileLinkResource(id, source);
	checkResource(resource);
	
	assertNotNull("The content should be not be null.", resource.getContent());
	assertUndefined("The content should be undefined.", resource.getContent());
	
		// setContent should have no action
	var contentNode = resource.setContent(content);
	assertNotNull("The contentNode should be not be null.", contentNode);
	assertUndefined("The contentNode should be undefined.", contentNode);
	assertNotNull("The getContent should be not be null.", resource.getContent());
	assertUndefined("The getContent should be undefined.", resource.getContent());
	
	var element = resource.getElement();
	checkElement(element);
	
	assertEquals("element should have no children.", 0, element.childNodes.length);
}


/** Test - Load and Unload
 * Test loading the resource into the document.
 */
function testLoadUnload() {
	var count1 = document.getElementsByTagName(category).length;
	assertEquals("There should be no elements with the matching id", 0, countElements(category, id));
	
	var resource = new MozileLinkResource(id, source);
	resource.load();
	var count2 = document.getElementsByTagName(category).length;
	assertEquals("There should be one more element now.", count2, count1+1);
	assertEquals("There should be exactly one element with the matching id", 1, countElements(category, id));
	
	resource.unload();
	var count3 = document.getElementsByTagName(category).length;
	assertEquals("There should be one fewer element now.", count3, count2-1);
	assertEquals("There should be the same number of elements as when we started.", count3, count1);
	assertEquals("There should be no elements with the matching id", 0, countElements(category, id));
	
	resource.load();
	resource.unload(document.documentElement);
	var count4 = document.getElementsByTagName(category).length;
	assertEquals("There should be the same number of elements as when we started.", count4, count1);
	assertEquals("There should be no elements with the matching id", 0, countElements(category, id));
}





/** Check Link Resource
 * Check the particulars of a link resource.
 */
function checkLinkResource(resource) {
	assertEquals("getSource() should return the source", source, resource.getSource());
	assertEquals("getRelation() should return the defaultRelation", defaultRelation, resource.getRelation());
}


/** Check Link Resource NS
 * Check the particulars of a link resource.
 */
function checkLinkResourceNS(resource) {
	assertEquals("getSource() should return the source", source, resource.getSource());
	assertEquals("getRelation() should return the source", relation, resource.getRelation());
}


/** Check Link Element
 * Check the particulars of a link element.
 */
function checkLinkElement(element) {
	assertEquals("The href of element should equal source", source, element.getAttribute("href"));
	assertEquals("The rel of element should equal defaultRelation", defaultRelation, element.getAttribute("rel"));
}


/** Check Link Element NS
 * Check the particulars of a link element.
 */
function checkLinkElementNS(element) {
	assertEquals("The href of element should equal source", source, element.getAttribute("href"));
	assertEquals("The rel of element should equal relation", relation, element.getAttribute("rel"));
}

