/*
 * @fileoverview Tools for simulating UI inputs. Requires the use of element.focus(), so it will only work in Gecko 1.8+, i.e. Firefox 1.5+ and Seamonkey 1.0+.
 * @link http://mozile.mozdev.org 
 * @author James A. Overton <james@overton.ca>
 * @version 0.7.0
 */

// Make sure that the browser is supported.
var mozillaVersion = navigator.userAgent.match(/rv\:(\d+\.\d+)/)[1];
if(Number(mozillaVersion) < 1.8) throw Error("GUIUtils.js does not support Mozilla browsers before revision 1.8. This means Firefox 1.5+ and SeaMonkey 1.0+ are supported, earlier versions of Firefox and the Mozilla Suite are not.");

/*
 * Gets the first first parent which has the CSS property "-moz-user-foucs: normal".
 * @param {Node} node
 * @type Node
 */
function focusableParent(node) {
	while(node) {
		if(node.nodeType == node.ELEMENT_NODE) {
			if(document.defaultView.getComputedStyle(node, '').getPropertyValue("-moz-user-focus").toLowerCase() == "normal") return node;
		}
		node = node.parentNode;
	}
	return null;
}


/*
 * Simulates all the events fired by a mouse click: mousedown, focus, mouseup, click. Also sets the selection.
 * @param {Node} anchorNode The node to be clicked. The location of the mousedown event.
 * @param {Integer} anchorOffset Optional. The offset within the anchorNode. The default is 0.
 * @param {Node} focusNode Optional. The location of the mouseup event. Defaults to anchorNode.
 * @param {Integer} focusOffset Optional. The offset within the focusNode. The default is anchorOffset.
 * @type Void
 */
function click(anchorNode, anchorOffset, focusNode, focusOffset) {
	if(!anchorOffset) anchorOffset = 0;
	if(!focusNode) focusNode = anchorNode;
	if(!focusOffset) focusOffset = anchorOffset;
	var selection = window.getSelection();
	var range = document.createRange();
	
	var event = document.createEvent("MouseEvents");
	event.initMouseEvent("mousedown", true, true, window, 1, 0, 0, 0, 0, false, false, false, false, 1, null);
	anchorNode.dispatchEvent(event);
	
	range.setStart(anchorNode, anchorOffset);
	selection.removeAllRanges();
	selection.addRange(range);
	
	event = document.createEvent("Events");
	event.initEvent("focus", true, true);
	anchorNode.dispatchEvent(event);
	
	if(focusableParent(anchorNode)) focusableParent(anchorNode).focus();
	
	range.setEnd(focusNode, focusOffset);
	selection.removeAllRanges();
	selection.addRange(range);

	event = document.createEvent("MouseEvents");
	event.initMouseEvent("mouseup", true, true, window, 1, 0, 0, 0, 0, false, false, false, false, 1, null);
	focusNode.dispatchEvent(event);

	event = document.createEvent("MouseEvents");
	event.initMouseEvent("click", true, true, window, 1, 0, 0, 0, 0, false, false, false, false, 1, null);
	focusNode.dispatchEvent(event);
}


/*
 * Simulates all the events fired by a keypress: keydown, keypress, keyup.
 * @param {Integer} keyCode The code number for the key pressed. For alphanumeric keys it is 0.
 * @param {Integer} charCode The code number for the character the key represents. For non-alphanumeric keys it is 0.
 * @param {Boolean} ctrlKey Optional. Indicates whether the Control key was part of the key press.
 * @param {Boolean} altKey Optional. Indicates whether the Alt key was part of the key press.
 * @param {Boolean} shiftKey Optional. Indicates whether the Shift key was part of the key press.
 * @param {Boolean} metaKey Optional. Indicates whether the Meta key was part of the key press.
 * @type Void
 */
function presskey(keyCode, charCode, ctrlKey, altKey, shiftKey, metaKey) {
	if(!ctrlKey) ctrlKey = false;
	if(!altKey) altKey = false;
	if(!shiftKey) shiftKey = false;
	if(!metaKey) metaKey = false;

	var selection = window.getSelection();
	var element = selection.focusNode;
	if(element.nodeType != element.ELEMENT_NODE) element = element.parentNode;
	var event;

	event = document.createEvent("KeyEvents");
	event.initKeyEvent("keydown", true, true, document.defaultView, ctrlKey, altKey, shiftKey, metaKey, keyCode, charCode);
	element.dispatchEvent(event);
	
	event = document.createEvent("KeyEvents");
	event.initKeyEvent("keypress", true, true, document.defaultView, ctrlKey, altKey, shiftKey, metaKey, keyCode, charCode);
	element.dispatchEvent(event);
	
	event = document.createEvent("KeyEvents");
	event.initKeyEvent("keyup", true, true, document.defaultView, ctrlKey, altKey, shiftKey, metaKey, keyCode, charCode);
	element.dispatchEvent(event);
}

// TODO Clever KeyCode handling.
// http://www.xulplanet.com/references/objref/KeyboardEvent.html
function keycode(keyCode, ctrlKey, altKey, shiftKey, metaKey) {
	presskey(keyCode, 0, ctrlKey, altKey, shiftKey, metaKey);
}

/*
 * A convenience function which simulates typing a string of characters.
 * @param {String} characters A sequence of characters to be typed.
 * @type Void
 */
function type(characters) {
	for(var c=0; c < characters.length; c++){
		presskey(0, characters.charCodeAt(c));
	}
}

/*
 * A convenience function which simulates a keyboard accelerator (AKA keyboard shortcut). Uses Meta on Mac and Control on other platforms.
 * @param {String} character The single character for the accelerator.
 * @param {Boolean} altKey Optional. Indicates whether the Alt key was part of the key press.
 * @param {Boolean} shiftKey Optional. Indicates whether the Shift key was part of the key press.
 * @type Void
 */
function accelerator(character, altKey, shiftKey) {
	if(navigator.userAgent.toLowerCase().indexOf("macintosh") >= 0) presskey(0, character.charCodeAt(0), false, altKey, shiftKey, true);
	else presskey(0, character.charCodeAt(0), true, altKey, shiftKey, false);
}



function dumpEvents() {
	var events = new Array();
	var item, items;
	for(var e=0; e < mozileEventList.length; e++) {
		item = mozileEventList[e];
		if(item.type!="focus") continue;
		items = new Array();
		for(i in item) {
			items.push(i+"="+item[i]);
		}
		events.push(items.join(", "));
	}
	return events.join("\n");
}