/** Mozile Resource Test Script
 * @fileoverview This file contains JSUnit test functions for testing the MozileResource class. 
 * Requires MozileResourcesUtils.js
 * 
 * @link http://mozile.mozdev.org 
 * @author James A. Overton <james@overton.ca>
 * @version 0.7.2
 */



var category = "script";
var id = "testScript";
var defaultType = "";
var type = "text/javascript";
var defaultNamespace = XHTMLNS;
var namespace = "http://namespace.com/";
var content = "Test content.";
	
/** Test - Constructor
 * Make sure that the MozileResource constructor is working. Use the default namespace.
 */
function testConstructor() {
	var resource = new MozileResource(category, id);
	checkResource(resource);
}



/** Test - Constructor NS
 * Make sure that the MozileResource constructor is working. Define a namespace this time.
 */
function testConstructorNS() {
	var resource = new MozileResource(category, id, type, namespace);
	checkResourceNS(resource);
}



/** Test - Create Element
 * Create an element using the default namespace.
 */
function testCreateElement() {
	var resource = new MozileResource(category, id);

		// getElement creates a new element	
	var element1 = resource.getElement();
	checkElement(element1);
	assertEquals("element1 should have exactly no children.", 0, element1.childNodes.length);

		// createElement creates a new element to replace the old one
	var element2 = resource.createElement();
	checkElement(element2);

		// getElement should get element2
	var element3 = resource.getElement();
	checkElement(element3);
	
	compareElements(element1, element2, element3);
}


/** Test - Create Element NS
 * Create an element, specifying a namespace.
 */
function testCreateElementNS() {
	var resource = new MozileResource(category, id, type, namespace);

		// getElement creates a new element	
	var element1 = resource.getElement();
	checkElementNS(element1);

		// createElement creates a new element to replace the old one
	var element2 = resource.createElement();
	checkElementNS(element2);

		// getElement should get element2
	var element3 = resource.getElement();
	checkElementNS(element3);
	
	compareElements(element1, element2, element3);
}


/** Test - Content
 * Test the content methods.
 */
function testContent() {
	var resource = new MozileResource(category, id);
	checkResource(resource);
	
	assertNotNull("The content should be not be null.", resource.getContent());
	assertUndefined("The content should be undefined.", resource.getContent());

	
	var textNode = document.createTextNode(content);
	var contentNode = resource.setContent(content);
	assertEquals("setContent should return a text node.", textNode.nodeType, contentNode.nodeType);
	assertEquals("getContent should return a text node.", textNode.nodeType, resource.getContent().nodeType);
	assertTrue("contentNode and getContent() should be identical", contentNode === resource.getContent());
	
	assertEquals("getContent().textContent should be content.", content, resource.getContent().textContent);
	
	var element = resource.getElement();
	checkElement(element);
	
	assertEquals("element should have exactly one child.", 1, element.childNodes.length);
	assertTrue("element.firstChild shouldbe identical to contentNode", element.firstChild === contentNode);
	assertEquals("element.firstChild.textContent should equal content", content, element.firstChild.textContent);
}


/** Test - Load and Unload
 * Test loading the resource into the document.
 */
function testLoadUnload() {
	var count1 = document.getElementsByTagName(category).length;
	assertEquals("There should be no elements with the matching id", 0, countElements(category, id));
	
	var resource = new MozileResource(category, id);
	resource.load();
	var count2 = document.getElementsByTagName(category).length;
	assertEquals("There should be one more element now.", count2, count1+1);
	assertEquals("There should be exactly one element with the matching id", 1, countElements(category, id));
	
		// Loading again should have no effect
	resource.load();
	var count2a = document.getElementsByTagName(category).length;
	assertEquals("There should be one more element now.", count2a, count1+1);
	assertEquals("There should be exactly one element with the matching id", 1, countElements(category, id));
	
	resource.unload();
	var count3 = document.getElementsByTagName(category).length;
	assertEquals("There should be one fewer element now.", count3, count2-1);
	assertEquals("There should be the same number of elements as when we started.", count3, count1);
	assertEquals("There should be no elements with the matching id", 0, countElements(category, id));
	
		// Test unloading from an element
	resource.load();
	resource.unload(document.documentElement);
	var count4 = document.getElementsByTagName(category).length;
	assertEquals("There should be the same number of elements as when we started.", count4, count1);
	assertEquals("There should be no elements with the matching id", 0, countElements(category, id));
	
		// Make sure that only one script is being unloaded at a time
	var resource2 = new MozileResource(category, id+"2");
	resource.load();
	resource2.load();
	
	var count5 = document.getElementsByTagName(category).length;
	assertEquals("There should be two elements more than when we started.", count5, count1+2);
	
	resource2.unload()
	var count6 = document.getElementsByTagName(category).length;
	assertEquals("There should be one element more than when we started.", count6, count1+1);
	
	resource.unload();
}


