/** Mozile Style Resource Test Style
 * @fileoverview This file contains JSUnit test functions for testing the MozileStyleResource class. 
 * Requires MozileResourcesUtils.js
 * 
 * @link http://mozile.mozdev.org 
 * @author James A. Overton <james@overton.ca>
 * @version 0.7.2
 */


var category = "style";
var id = "testStyle";
var defaultType = "text/css";
var type = "text/CSS";
var defaultMedia = "";
var media = "screen";
var defaultNamespace = XHTMLNS;
var namespace = "http://namespace.com/";
var content = "Test content.";


/** Test - Constructor
 * Make sure that the MozileStyleResource constructor is working. Use the default namespace.
 */
function testConstructor() {
	var resource = new MozileStyleResource(id);
	checkResource(resource);
	checkStyleResource(resource);
}


/** Test - Constructor
 * Make sure that the MozileStyleResource constructor is working. Define a type and a namespace.
 */
function testConstructorNS() {
	var resource = new MozileStyleResource(id, type, media, namespace);
	checkResourceNS(resource);
	checkStyleResourceNS(resource);
}


/** Test - Create Element
 * Create an element using the default namespace.
 */
function testCreateElement() {
	var resource = new MozileStyleResource(id);

		// getElement creates a new element	
	var element1 = resource.getElement();
	checkElement(element1);
	checkStyleElement(element1);

		// createElement creates a new element to replace the old one
	var element2 = resource.createElement();
	checkElement(element2);
	checkStyleElement(element2);

		// getElement should get element2
	var element3 = resource.getElement();
	checkElement(element3);
	checkStyleElement(element3);
	
	compareElements(element1, element2, element3);
}


/** Test - Create Element
 * Create an element, specifying a type and a namespace.
 */
function testCreateElementNS() {
	var resource = new MozileStyleResource(id, type, media, namespace);

		// getElement creates a new element	
	var element1 = resource.getElement();
	checkElementNS(element1);
	checkStyleElementNS(element1);

		// createElement creates a new element to replace the old one
	var element2 = resource.createElement();
	checkElementNS(element2);
	checkStyleElementNS(element2);

		// getElement should get element2
	var element3 = resource.getElement();
	checkElementNS(element3);
	checkStyleElementNS(element3);
	
	compareElements(element1, element2, element3);
}


/** Test - Content
 * Test the content methods.
 */
function testContent() {
	var resource = new MozileStyleResource(id);
	checkResource(resource);
	
	assertNotNull("The content should be not be null.", resource.getContent());
	assertUndefined("The content should be undefined.", resource.getContent());
	
	var textNode = document.createTextNode(content);
	var contentNode = resource.setContent(content);
	assertEquals("setContent should return a text node.", textNode.nodeType, contentNode.nodeType);
	assertEquals("getContent should return a text node.", textNode.nodeType, resource.getContent().nodeType);
	assertTrue("contentNode and getContent() should be identical", contentNode === resource.getContent());
	
	assertEquals("getContent().textContent should be content.", content, resource.getContent().textContent);
	
	var element = resource.getElement();
	checkElement(element);
	
	assertEquals("element should have exactly one child.", 1, element.childNodes.length);
	assertTrue("element.firstChild shouldbe identical to contentNode", element.firstChild === contentNode);
	assertEquals("element.firstChild.textContent should equal content", content, element.firstChild.textContent);
}


/** Test - Load and Unload
 * Test loading the resource into the document.
 */
function testLoadUnload() {
	var count1 = document.getElementsByTagName(category).length;
	assertEquals("There should be no elements with the matching id", 0, countElements(category, id));
	
	var resource = new MozileStyleResource(id);
	resource.load();
	var count2 = document.getElementsByTagName(category).length;
	assertEquals("There should be one more element now.", count2, count1+1);
	assertEquals("There should be exactly one element with the matching id", 1, countElements(category, id));
	
	resource.unload();
	var count3 = document.getElementsByTagName(category).length;
	assertEquals("There should be one fewer element now.", count3, count2-1);
	assertEquals("There should be the same number of elements as when we started.", count3, count1);
	assertEquals("There should be no elements with the matching id", 0, countElements(category, id));
	
	resource.load();
	resource.unload(document.documentElement);
	var count4 = document.getElementsByTagName(category).length;
	assertEquals("There should be the same number of elements as when we started.", count4, count1);
	assertEquals("There should be no elements with the matching id", 0, countElements(category, id));
}


/** Test - Style Sheet
 * Test the getStylesheet method.
 */
function testStylesheet() {
	var resource = new MozileStyleResource(id);
	checkResource(resource);
	resource.unload();
	
	var undefinedSheet = resource.getStylesheet();
	assertNotNull("The undefinedSheet should not be null", stylesheet);
	assertUndefined("The undefinedSheet should be undefined", undefinedSheet);
	
	resource.load();

	var stylesheet = resource.getStylesheet();
	assertNotNull("The stylesheet should not be null", stylesheet);
	assertNotUndefined("The stylesheet should not be undefined", stylesheet);
	
	assertEquals("There should be exactly 0 rules", 0, stylesheet.cssRules.length);
	stylesheet.insertRule("#dummyRule { color: white; }", 0);
	assertEquals("There should be exactly 1 rule", 1, stylesheet.cssRules.length);
	
	resource.unload();
}




/** Check Style Resource
 * Check the particulars of a link resource.
 */
function checkStyleResource(resource) {
	assertEquals("getMedia() should return the defaultMedia", defaultMedia, resource.getMedia());
}


/** Check Style Resource NS
 * Check the particulars of a link resource.
 */
function checkStyleResourceNS(resource) {
	assertEquals("getMedia() should return the source", media, resource.getMedia());
}


/** Check Style Element
 * Check the particulars of a link element.
 */
function checkStyleElement(element) {
	assertNull("There should be no media attribute", element.getAttribute("media"));
}


/** Check Style Element NS
 * Check the particulars of a link element.
 */
function checkStyleElementNS(element) {
	assertEquals("The media of element should equal media", media, element.getAttribute("media"));
}

