/* ***** BEGIN LICENSE BLOCK *****
 * Licensed under Version: MPL 1.1/GPL 2.0/LGPL 2.1
 * Full Terms at http://mozile.mozdev.org/license.html
 *
 * Software distributed under the License is distributed on an "AS IS" basis,
 * WITHOUT WARRANTY OF ANY KIND, either express or implied. See the License
 * for the specific language governing rights and limitations under the
 * License.
 *
 * The Original Code is Playsophy code (www.playsophy.com).
 *
 * The Initial Developer of the Original Code is Playsophy
 * Portions created by the Initial Developer are Copyright (C) 2002-2003
 * the Initial Developer. All Rights Reserved.
 *
 * Contributor(s):
 *	David Palm <dvdplm@iperbole.bologna.it>
 *	James A. Overton <james@overton.ca>
 *
 * ***** END LICENSE BLOCK ***** */

/** Copy Cut Paste Module
 * @fileoverview Allows Mozile to do some basic Copy&Paste operations. In an unprivileged environment it will use a local clipboard. In a privileged enviromnent the system clipboard will be used.
 * @link http://mozile.mozdev.org 
 * @author David Palm <dvdplm@iperbole.bologna.it>
 * @version 0.7
 */

// A string which contains the local clipboard contents
mozile.getModule("CopyCutPaste").clipboard = "";


/**
 * Gets the local or the global clipboard, depending on privileges.
 * @type String
 */
mozile.getModule("CopyCutPaste").getClipboard = function() {
	var result = "";
	
	if(mozile.isEnhanced("getClipboard")) {
		mozile.setSharedData("clipboard", "");
		mozile.clientRequest("getClipboard", "if(mozile.getSharedData('clipboard')) mozileEditor.insertString(mozile.getSharedData('clipboard'))");
		return result;
	}

		// If the "requestPermissions" option has not been set to "true", then use the document's clipboard and not the system clipboard.
	else if(!mozile.isExtension() && !mozile.getModule("CopyCutPaste").getOption("requestPermissions") ) {
		if(this.clipboard) result = this.clipboard;
	}
		// Otherwise use the system clipboard.
	else {
			//Get privileges
		try{netscape.security.PrivilegeManager.enablePrivilege("UniversalXPConnect UniversalBrowserWrite ");}
		catch(e){ return result; }
		var clip = Components.classes["@mozilla.org/widget/clipboard;1"].createInstance(Components.interfaces.nsIClipboard);
		if (!clip) return result;
		var trans = Components.classes["@mozilla.org/widget/transferable;1"].createInstance(Components.interfaces.nsITransferable);
		if (!trans) return result;
		trans.addDataFlavor("text/unicode");
		clip.getData(trans,clip.kGlobalClipboard);
		var str = new Object();
		var strLength	=	new Object();
		trans.getTransferData("text/unicode",str,strLength);
		if(str) str = str.value.QueryInterface(Components.interfaces.nsISupportsString);
		else return result;
		if (str) result = str.data.substring(0,strLength.value / 2);
	}
	
	return result;
}


/**
 * Sets the local clipboard. The global clipboard will be set automatically if there was a keyboard event. If this method is privilieged, it will set the global clipboard in any case.
 * @param Sting content
 * @type String
 */
mozile.getModule("CopyCutPaste").setClipboard = function(content) {
	
	 // Share the data with the Mozile Extension
	if(mozile.isEnhanced("setClipboard")) {
		mozile.setSharedData("clipboard", content);
		mozile.setSharedData("clientRequest", "setClipboard");
	}
	
	// If the "requestPermissions" option has not been set to "true", then use the document's clipboard and not the system clipboard.
	if(!mozile.isExtension() && !mozile.getModule("CopyCutPaste").getOption("requestPermissions") ) {
		this.clipboard = content;
	}
		// Otherwise use the system clipboard.
	else {
		//Get privileges
		try{netscape.security.PrivilegeManager.enablePrivilege("UniversalXPConnect UniversalBrowserWrite ");}
		catch(e){	return "";	}
		const	gClipboardHelper	=	Components.classes["@mozilla.org/widget/clipboardhelper;1"].getService(Components.interfaces.nsIClipboardHelper);
		if(gClipboardHelper) gClipboardHelper.copyString(content);
		else this.clipboard = content;
	}
	
	
	return content;
}


/** 
 * Copy the current selection to system clipboard. Makes use of the nsIClipboardHelper interface and is not a "full-scale" copy-to-clipboard method; this means the selection will be copied as a unicode string and all formatting will be stripped (try copying an unordered list and watch how gecko turns it into a string with newlines, tabs and "*" instead of bullets :-)
 * @type String
 * @return The string that was copied to the clipboard when successful, false otherwise.
 */
mozile.getModule("CopyCutPaste").copy = function(){
	var	copyStr	=	mozile.getSelection().toString();
	this.setClipboard(copyStr);
	return copyStr;
}

/** 
 * Cut the current selection to system clipboard. Calls Mozile.copy to do the copy then deletes the current selection (using MozileCore.js's deleteContents() method)
 * @type String
 * @return The string that was cut to the clipboard when successful, false otherwise.
 */
mozile.getModule("CopyCutPaste").cut = function(){
	var copyStr  = this.copy() ;
	if(copyStr){
		mozile.getSelection().deleteContents();
		return copyStr;
	}
	else{
		mozile.debug(f,0,"Cut command failed. Maybe a problem in copy()?");
		return false;
	}
}
/** 
 * Paste from clipboard, replacing selectin if any. The implementation uses the full nsIClipboard interface. Should probably be improved/extended to handle at least URLs.
 * @type String
 * @return The string that was pasted when successful, false otherwise.
 */
mozile.getModule("CopyCutPaste").paste = function(){
	var clipboard = this.getClipboard();
	if(clipboard && clipboard != "") mozileEditor.insertString(clipboard);
	return clipboard;	
}


/**
 * Initialize the module.
 * @type Void
 */
mozile.getModule("CopyCutPaste").init = function(){
	var mozileCopy = mozile.getCommandList().createCommand("MozileCommand: id=Mozile-Copy, label=Copy, tooltip='Copy selection', accelerator='Command-C', image='"+mozile.getRoot()+"images/copy.png'");
	mozileCopy.execute = function() {
		mozile.getModule("CopyCutPaste").copy();
		return true;
	}
	var mozileCut = mozile.getCommandList().createCommand("MozileCommand: id=Mozile-Cut, label=Cut, tooltip='Cut selection', accelerator='Command-X', image='"+mozile.getRoot()+"images/cut.png'");
	mozileCut.execute = function() {
		mozile.getModule("CopyCutPaste").cut();
		return true;
	}
	var mozilePaste = mozile.getCommandList().createCommand("MozileCommand: id=Mozile-Paste, label=Paste, tooltip='Paste selection', accelerator='Command-V', image='"+mozile.getRoot()+"images/paste.png'");
	mozilePaste.execute = function() {
		mozile.getModule("CopyCutPaste").paste();
		return true;
	}
	
	mozile.getCommandList().createCommand("MozileCommandSeparator: id=Mozile-CopySeparator");
}


//Set up copy, cut and paste commands
if(mozile.getCommandList()) {
	mozile.getModule("CopyCutPaste").init();
}
