/** Mozile Module Test Script 1
 * @fileoverview This file contains JSUnit test functions for testing Mozile's module loading and unloading methods. It is the first of three such scripts. 
 * 
 * @link http://mozile.mozdev.org 
 * @author James A. Overton <james@overton.ca>
 * @version 0.7.0
 */

// create the Mozile instance, absolutely no debugging messages
var mozile, scripts, j, i;


/** Test - Mozile Constructor
 * Make sure that the Mozile object constructor has functioned properly.
 */
function mozileConstructor() {
	var root = "../../";
	var mode = "HTML";
	var namespace = "namespace";
	var debugLevel = "-1";
	mozile = new Mozile("root='"+root+"', mode="+mode+", namespace='"+namespace+"', debugLevel="+debugLevel+", warnBeforeUnload=false");
	assertNotNull("The mozile object should exist", mozile);
	assertNotUndefined("The mozile object should be defined.", mozile);
	assertEquals("The root should be "+root, root, mozile.root);
	assertEquals("The mode should be "+mode, mode, mozile.mode);
	assertEquals("The namespace should be "+namespace, namespace, mozile.namespace);
	assertEquals("The debugLevel should be "+debugLevel, debugLevel, mozile.debugLevel);
}



/** Test - Registration
 * Test the module registration methods. A module must be loaded, and in the moduleList, before it can be registered. This includes registering a module, making sure it is registered, and the unregistering it and making sure it is no longer registered.
 */
function registration() {
	var testModule = "TestModule";
	var otherModule = "otherModule";
	var version = "1.2.3";
	
	// Fake loading the module by adding an entry to the mozile.moduleList.
	mozile.moduleList[testModule] = new Array();
	
	// Register the Test Module, and check to make sure that it worked.
	var result = mozile.registerModule(testModule, version);	
	assertTrue("The registerModule function should return true", result);
	assertNotUndefined("There should be now be a "+ testModule +" entry in the moduleList", mozile.moduleList[testModule]);
	assertEquals("The "+ testModule +" entry should have a value "+version, version, mozile.moduleList[testModule]['version']);

	// Make sure that the test module is loaded, and that we aren't getting false positives.	
	assertTrue("Now the "+ testModule +" should be loaded", mozile.moduleIsLoaded(testModule));
	assertFalse("But the "+ otherModule +" should not be loaded", mozile.moduleIsLoaded(otherModule));
	
	// Unregister the moudles, and check that it is no longer loaded.
	result = mozile.unregisterModule(testModule);
	assertTrue("The unregisterModule function should succeed with "+ testModule, result);
	assertFalse("Now the "+ testModule +" should not be loaded", mozile.moduleIsLoaded(testModule));

	// Try to unregister a module which has not been registered.
	result = mozile.unregisterModule(otherModule);
	assertFalse("The unregisterModule function should fail with "+ otherModule, result);

}


/** Test - Check Requirements
 * Check to see that the checkRequirements method is working properly, allowing version strings which should be allowed, and not allowing ones which should not be allowed.
 */
function checkRequirements() {

	var requirements = "requireVersion=1.2.3";
	assertTrue("The requirements for 1.2.3 should be met", mozile.checkRequirements("1.2.3",requirements));
	assertFalse("The requirements for 1.2.4 should not be met", mozile.checkRequirements("1.2.4",requirements));

	requirements = "minVersion=1.0.0, maxVersion=1.3.4, notVersion=1.2.1, notVersion=1.2.3";
	var checks = new Array();
	checks["0"]=false;
	checks[""]=false;
	checks["cats"]=false;
	checks["0.9.9"]=false;
	checks["0.9.99"]=false;
	checks["0.99.99"]=false;
	checks["1.0.0"]=true;
	checks["1.1.0"]=true;
	checks["1.1.1"]=true;
	checks["1.1.100"]=true;
	checks["1.2.0"]=true;
	checks["1.2.1"]=false;
	checks["1.2.2"]=true;
	checks["1.2.3"]=false;
	checks["1.3.0"]=true;
	checks["1.3.4"]=true;
	checks["1.3.5"]=false;
	checks["1.33.5"]=false;
	checks["1.4.5"]=false;
	checks["2.0.0"]=false;
	checks["200.0.0"]=false;

	// Loop through the array, making sure each entry matches the specified value.
	for(key in checks) {
		if(checks[key]) {
			assertTrue("The requirements for "+key+" should be met", mozile.checkRequirements(key,requirements));
		}
		else {
			assertFalse("The requirements for "+key+" should not be met", mozile.checkRequirements(key,requirements));
		}
	
	}
}




