/* ***** BEGIN LICENSE BLOCK *****
 * Version: MPL 1.1/GPL 2.0/LGPL 2.1
 *
 * The contents of this file are subject to the Mozilla Public License Version
 * 1.1 (the "License"); you may not use this file except in compliance with
 * the License. You may obtain a copy of the License at
 * http://www.mozilla.org/MPL/
 *
 * Software distributed under the License is distributed on an "AS IS" basis,
 * WITHOUT WARRANTY OF ANY KIND, either express or implied. See the License
 * for the specific language governing rights and limitations under the
 * License.
 *
 * The Original Code is James A. Overton's code (james@overton.ca).
 *
 * The Initial Developer of the Original Code is James A. Overton.
 * Portions created by the Initial Developer are Copyright (C) 2005
 * the Initial Developer. All Rights Reserved.
 *
 * Contributor(s):
 *
 * Alternatively, the contents of this file may be used under the terms of
 * either the GNU General Public License Version 2 or later (the "GPL"), or
 * the GNU Lesser General Public License Version 2.1 or later (the "LGPL"),
 * in which case the provisions of the GPL or the LGPL are applicable instead
 * of those above. If you wish to allow use of your version of this file only
 * under the terms of either the GPL or the LGPL, and not to allow others to
 * use your version of this file under the terms of the MPL, indicate your
 * decision by deleting the provisions above and replace them with the notice
 * and other provisions required by the GPL or the LGPL. If you do not delete
 * the provisions above, a recipient may use your version of this file under
 * the terms of any one of the MPL, the GPL or the LGPL.
 *
 * ***** END LICENSE BLOCK ***** */







// A global variable containing the current target window.
var mozileTarget;

/** Mozile - Open -
 * Opens a new window from the current Mozile document. If this code is executed from inside the Mozile extension, then the focused window is used to open the new window. Otherwise, the current window is used.
 * TODO: This function is probably not used any more.
 * 
 * @param url String. URL to open. 
 * @param name Optional String. Name of the window to open.
 * @param options Optional String. Window options to use.
 * @return The Window obbject just opened, if successful.
 */
Mozile.prototype.open = function(url) {
	var name = "";
	var options = "";
	
	if(arguments.length == 2) name = arguments[2];
	if(arguments.length == 3) options = arguments[3];
	
	if(mozile.extension) {
		mozileTarget = document.commandDispatcher.focusedWindow;
	}
	else {
		mozileTarget = window;
	}
	return window.open(url,name,options);
}















/** Mozile - Create Toolbar -
 * Does all the work required to add the Mozile toolbar to the current document. This includes adding a new style sheet to the document, and adding a CSS rule which associates an XBL binding with the body element (in the case of X/HTML), or a new <moziletoolbar/> element (which is inserted into an XML document).
 * 
 * @return True if successful.
 */
Mozile.prototype.createToolbar = function() {
	var f = new Array();
	f["File"] = "core/interface.js";
	f["Function"] = "Mozile.createToolbar()";
	this.debug(f,1,"Creating toolbar");

	// Add basic style-sheet to the document.
	this.loadLink(this.root + "core/interface.css", "Mozile-Core-interface.css");

	// Get the styleSheet
	if(!this.styleSheet) this.addStyleSheet();

	// Add a new rule to the end of the sheet which binds the toolbar to the body element.
	var rule;
	// Handle the X/HTML case, where the toolbar is attached to the body element.
	if(document.documentElement.tagName.toLowerCase()=="html") {
		rule = "body { -moz-binding: url(" + this.root +"core/interface.xml#toolbar); }";
	}
	// Handle the XML case, where the toolbar is attached to a new moziletoolbar element, inserted near the top of the document.
	else {
		rule = "moziletoolbar { -moz-binding: url(" + this.root +"core/interface.xml#toolbar); }";
		document.documentElement.insertBefore(document.createElement("moziletoolbar"), document.documentElement.firstChild);
	}
	
	// Insert the rule.
	this.styleSheet.insertRule(rule, this.styleSheet.cssRules.length);

	// Create base command list object, and populate it.
	this.rootCommandList = mozile.createCommand("MozileCommandList: id=Mozile-RootList, label='Mozile Root List'");

	// Create the basic Mozile command list.
	var mozileList = this.rootCommandList.createCommand("MozileCommandList: id=Mozile-firstList, label='Mozile First List', image='"+this.root+"images/Mozile-Smaller.png', buttonPosition=0, menuPosition=0");
	
	var about = mozileList.createCommand("MozileCommand: id=Mozile-About, label='About Mozile'");
	about.command = function(event) {
		//window.open(mozile.root+"core/about.xul", "About Mozile", "centerscreen,chrome,resizable=yes");
		mozile.showAboutInterface();
		document.getAnonymousElementByAttribute(mozile.aboutInterface.parentNode, "id", "version").value = mozile.version;
	}
	
	var accel = mozileList.createCommand("MozileCommand: id=Mozile-Accelerators, label='Keyboard Shortcuts'");
	accel.command = function(event) {
		if(mozile.keyboardShortcuts) {
			var message = "Mozile defines the following keyboard shortcuts:";
			var accels = mozile.acceleratorList;
			for(key in accels) {
				message = message +"\n"+accels[key].label+" => "+key;
			}
			alert(message);
		} 
		else {
			alert("Mozile keyboard shortcuts have been turned off.");
		}
	}
	
	var debug = mozileList.createCommand("MozileCommand: id=Mozile-Debug, label=Debug");
	debug.command = function(event) {
		//mozile.open(mozile.root+"core/debug.xul", "Mozile Debugging", "centerscreen,chrome,resizable=yes");g
		mozile.showMessageInterface();
		mozileDebugInit();
	}
	
	//var test = mozileList.createCommand("MozileCommand: id=Mozile-Test, label='Testing Tools'");
	//test.command = function(event) {
	//	window.open(mozile.root+"testing/index.html", "Mozile Testing", "");
	//}
	
	var report = mozileList.createCommand("MozileCommand: id=Mozile-ReportBugs, label='Report a Bug'");
	report.command = function(event) {
		window.open("http://mozile.mozdev.org/bugs.html", "Mozile Bugs", "");
	}
	
	var home = mozileList.createCommand("MozileCommand: id=Mozile-Home, label='Mozile Home'");
	home.command = function(event) {
		window.open("http://mozile.mozdev.org", "Mozile Home", "");
	}
	
	var help = mozileList.createCommand("MozileCommand: id=Mozile-Help, label='Help'");
	help.command = function(event) {
		window.open(mozile.root+"docs/index.html", "Mozile Help", "");
	}

	var save = this.rootCommandList.createCommand("MozileCommand: id=Mozile-Save, label=Save, tooltip='Save to a dialog', accelerator='Command-S', image='"+this.root+"images/filesave.png'");
	save.command = function(event) {
		mozile.save();
	}

	var saveAs = this.rootCommandList.createCommand("MozileCommand: id=Mozile-SaveAs, label='Save As', tooltip='Save as...', accelerator='Command-Shift-S', image='"+this.root+"images/filesaveas.png'");
	saveAs.command = function(event) {
		mozile.saveAs();
	}

/*
	var test = this.rootCommandList.createCommand("MozileCommand: id=Mozile-Test, label=Test, tooltip='Test', image='"+this.root+"images/view-source.png'");
	test.command = function(event) {
		var f = new Array();
		f["File"] = "core/interface.js";
		f["Function"] = "test.command";
	
		//mozile.openDialog(mozile.root+"core/save.xul", "save", "modal,dialog,centerscreen,chrome,resizable=yes,height=300,width=400");
		
		//mozile.openDialog(mozile.root+"core/save.xul", "save", "modal,dialog,centerscreen,chrome,resizable=yes,height=300,width=400");
		
		//mozileSaveInit();
		
		alert(window.pageYOffset +" "+ window.innerHeight +" "+ window.scrollMaxY);
		
		//document.location = "mailto:?subject="+document.title+"&body="+ escape("\n\n\n\n"+mozile.documentToXML());
		
		//mozile.status(f,4,"Contents: "+printXML(document));
	}
*/

	return true;
}





/** Mozile - Initialize Toolbar -
 * Does some basic setup on the toolbar, which can't be done until after all modules have loaded. 
 * TODO: Make the basic toolbar useful, with a Help and About command.
 * 
 * @return True if successful.
 */
Mozile.prototype.initializeToolbar = function() {
	var f = new Array();
	f["File"] = "core/interface.js";
	f["Function"] = "Mozile.initializeToolbar()";
	this.debug(f,1,"Initializing toolbar");

	// Create the command box.
	this.rootCommandList.createBox();
	
	// Add the rootCommandList.box to the toolbar
	this.toolbar.appendChild(this.rootCommandList.box);
	
	return true;

}







/** Mozile - Hide Toolbar -
 * Hides the Mozile toolbar.
 * 
 * @return True if successful.
 */
Mozile.prototype.hideToolbar = function() {
	var f = new Array();
	f["File"] = "core/interface.js";
	f["Function"] = "Mozile.hideToolbar()";
	this.debug(f,1,"Hiding toolbar");

	this.toolbar.collapsed = true;
	this.statusbar.collapsed = true;
	
	return true;
}




/** Mozile - Show Toolbar -
 * Shows (unhides) the Mozile toolbar. If this is the first time that the toolbar has been shown, then it calls this.initializeToolbar().
 * 
 * @return True if successful.
 */
Mozile.prototype.showToolbar = function() {
	var f = new Array();
	f["File"] = "core/interface.js";
	f["Function"] = "Mozile.showToolbar()";
	this.debug(f,1,"Showing toolbar. First time? " + this.firstToolbarShow);
	
	if(this.firstToolbarShow) {
		if(this.toolbarPosition == "absolute") {
			this.toolbar.style.position = "absolute";
			this.statusbar.style.position = "absolute";
		}
		if(this.toolbarUpdateFrequency == 2) {
			window.setInterval("mozile.moveToolbar()", 100);
		}
		this.initializeToolbar();
		this.storeState("Initial state");
		this.firstToolbarShow=false;
	}
	this.toolbar.collapsed = false;
	this.statusbar.collapsed = false;

	return true;
}



/** Mozile - Move Toolbar -
 * Centres the Mozile toolbar and statusbar in the window, by calculating the proper size of the CSS "left" property.
 * 
 * @return True if successful.
 */
Mozile.prototype.moveToolbar = function() {
	var f = new Array();
	f["File"] = "core/interface.js";
	f["Function"] = "Mozile.moveToolbar()";
	//this.debug(f,1,"Moving toolbar.");
	
	if(this.toolbar == null) return false;
	
	if(!this.lastWindowSize) {
		this.lastWindowSize = new Array();
	}	
	
	if(this.lastWindowSize["innerWidth"] == window.innerWidth && 
		this.lastWindowSize["innerHeight"] == window.innerHeight &&
		this.lastWindowSize["pageXOffset"] == window.pageXOffset &&
		this.lastWindowSize["pageYOffset"] == window.pageYOffset &&
		this.lastWindowSize["toolbarWidth"] == this.toolbar.boxObject.width) {
		return true;
	}
	
	this.debug(f,1,"Moving toolbar.");
	
	// This seems unnecessary, but it makes the this.toolbar.boxObject.width read properly (or at least better).
	this.toolbar.style.left= "0px";

	// Set the left position so the toolbars are centred.
	this.toolbar.style.left = (window.innerWidth - this.toolbar.boxObject.width) / 2 +"px";
	this.statusbar.style.left = (window.innerWidth - this.statusbar.boxObject.width) / 2 +"px";
	
	// If the positioning type is "absolute" set the top position for the toolbars.
	if(this.toolbar.style.position == "absolute") {
		this.debug(f,1,"Moving toolbars");
		this.toolbar.style.top = window.pageYOffset - 1 +"px";
		this.statusbar.style.top = window.pageYOffset + window.innerHeight - this.statusbar.boxObject.height + 1 +"px";	
	}

	// Store the current size and position.
	this.lastWindowSize["innerWidth"] = window.innerWidth;
	this.lastWindowSize["innerHeight"] = window.innerHeight;
	this.lastWindowSize["pageXOffset"] = window.pageXOffset;
	this.lastWindowSize["pageYOffset"] = window.pageYOffset;
	this.lastWindowSize["toolbarWidth"] = this.toolbar.boxObject.width;
	
	//this.debug(f,2, window.innerWidth +" "+ this.toolbar.style.left +"/"+ this.toolbar.boxObject.width +" "+ this.statusbar.style.left +"/"+ this.statusbar.boxObject.width);

	return true;
}




/** Mozile - Update Toolbar -
 * Updates all of the commands in the commandList so that they will reflect the current selection.
 * 
 * @param force Optional Boolean, forces update when "true" and toolbarUpdateFrequency > 0.
 * @return True if successful.
 */
Mozile.prototype.updateToolbar = function() {
	var f = new Array();
	f["File"] = "core/interface.js";
	f["Function"] = "Mozile.updateToolbar()";
	this.debug(f,1,"Updating toolbar");
	
	if(this.toolbarUpdateFrequency==0) return true;
	
	var force = false;
	if(arguments.length > 0) {
		force = arguments[0];
	}
	
	var selection = mozile.getSelection();
	
	// If the update is not being forced, then check to see if it is required.
	if(!force) {
		// Don't update if the node has not changed.
		if(selection.focusNode == this.lastFocusNode) {
			this.debug(f,0,"Toolbar update not required");
			return true;
		}	
	}
	
	this.debug(f,1,"Updating toolbar");
	this.lastFocusNode = selection.focusNode;
	
	// Update all the commands in the command list
	for(var command in this.commandList) {
		try {
			this.commandList[command].update();
		} catch(e) {
			//alert("Error "+e+"\n"+command);
		}
		
	}
	
	this.moveToolbar();
	
	return true;
}






/** Mozile Save To Dialog -
 * This is the most basic Mozile save method. Opens a new window which will grab the serialized source of the document and display it for the user.
 * 
 * @return Always true.
 */
Mozile.prototype.saveToDialog = function() {
	var f = new Array();
	f["File"] = "core/interface.js";
	f["Function"] = "mozileSaveToDialog";
	
	//this.open(mozile.root+"core/source.xul", "Mozile Save Dialog", "centerscreen,chrome,resizable=yes");
	
	this.showSourceInterface();
	document.getAnonymousElementByAttribute(this.sourceInterface.parentNode,'id','sourceText').value = this.content();
	
	this.status(f,1,"Displaying document contents.");
	return true;
}



/** Mozile - Save -
 * This function sets the Mozile.saveConfig property and calls the appropriate save function from Mozile.saveList: either "default" or "custom". If neither exists, then it opens the Save As dialog.
 * 
 * @return Always true.
 */
Mozile.prototype.save = function() {
	var mode;
	if(this.saveList["custom"] != null) {
		mode = "custom";
	}
	else {
		if(this.saveList["default"] != null) {
			mode = "default";
		}	
		else {
			return this.saveAs();
		}
	}

	//dumpArray(this.saveList[mode]);
	
	//this.saveConfig = this.saveList[mode];
	
	
	if(this.saveFlag == true || !this.saveConfig) {
		this.saveFlag = false;
		
		// Copy the config array by value (?)
		this.saveConfig = new Array();
		for(key in this.saveList[mode]) {
			this.saveConfig[key] = this.saveList[mode][key];
		}
	}

	//mozileSaveToDialog();
	
	eval( this.saveConfig["function"] + "()" );
	
	//alert(this.content());
	
	return true;
}




/** Mozile - Save As -
 * Opens Mozile's Save As dialog.
 * 
 * @return Always true.
 */
Mozile.prototype.saveAs = function() {

	this.showSaveInterface();
	mozileSaveInit();

	//var win = mozile.open(this.root+"core/save.xul", "save", "modal,dialog,centerscreen,chrome,resizable=yes,height=300,width=400");
	//win.focus();
	
	//mozileSaveInit();
	//mozile.openDialog();
	
	return true;
}












/** Mozile - Create Command -
 * This method creates a Mozile command object from a configuration string. It works almost identically to the MozileCommand.List.createCommand() method. However, this method is usually used only to create a root command list, which contains all the other commands for the Mozile toolbar. For this reason, it does not register commands.
 * 
 * Format (particular command objects may have other options): "MozileCommandObject: id=CommandId, label='Command Label', tooltip='Command Tool Tip', image='/path/to/image.file', accesskey=C, accelerator='Meta-C', buttonPosition=0, menuPosition=0"
 *
 * @param configString A configuration string, following the above format.
 * @return The newly created command object.
 */
Mozile.prototype.createCommand = function(configString) {
	var f = new Array();
	f["File"] = "core/interface.js";
	f["Function"] = "Mozile.createCommand()";
	this.debug(f,1,"Creating command "+ configString);

	// Parse the configuration string.
	var configArray = this.parseConfig(configString);
	
	var name = configArray['name'];
	var command;
	
	// Define the command creation string. I.e. "command = new MozileCommand(configArray)"
	var create = "command = new "+name+"(configArray)";
	
	// Evaluate the command creation string.
	eval(create);
	
	// This method does not register commands
	//this.registerCommand(command);
	
	return command;

}




/** Mozile - Register Command -
 * Registers a command or command list object, adding it to the this.commandList array.
 * 
 * @param command The command object.
 * @return True when successful, false otherwise.
 */
Mozile.prototype.registerCommand = function(command) {
	var f = new Array();
	f["File"] = "core/interface.js";
	f["Function"] = "Mozile.registerCommand()";
	this.debug(f,1,"Registering command "+ command +" with id "+ command.id);

	if(typeof(command) != "object") return false;
	
	var id = command.id;
	// If a command with that id is already registered, return false.
	if(id == null || this.commandIsRegistered(id)) return false;
	// Otherwise, add the command object to the commandList, with it id as the key.
	else {
		this.commandList[id] = command;
		if(command.accelerator) {
			var accel = command.accelerator;
			if(this.operatingSystem=="Mac") {
				accel = accel.replace("Command", "Meta");
			}
			else {
				accel = accel.replace("Command", "Control");
			}
			this.acceleratorList[accel] = command;
		}
		return true;
	}

}



/** Mozile - Unregister Command -
 * Unregisters a command or command list object by removing it from the commandList array.
 * 
 * @param id The id of the command. 
 * @return Always true.
 */
Mozile.prototype.unregisterCommand = function(id) {
	var f = new Array();
	f["File"] = "core/interface.js";
	f["Function"] = "Mozile.unregisterCommand()";
	this.debug(f,1,"Unregistering command "+ id);

	// If there is no such command, it can't be unregistered.
	if(!this.commandIsRegistered(id)) return false;
	else {
		this.commandList[id] = null;
		return true;
	}

}



/** Mozile - Command Is Registered -
 * Checks to see if a command with the given id has been registered.
 * 
 * @param id The id of the command. 
 * @return True if a command with that id is registered, false otherwise.
 */
Mozile.prototype.commandIsRegistered = function(id) {
	var f = new Array();
	f["File"] = "core/interface.js";
	f["Function"] = "Mozile.commandIsRegistered()";
	this.debug(f,1,"Checking for command "+ id);

	if(this.commandList[id]) return true;
	else return false;

}



/** Mozile - Execute Command -
 * Calls the command() method of the command object corresponding to the given id.
 * 
 * @param id The id of the node which triggered the command. 
 * @param event The event which triggered this command. 
 * @return Always true.
 */
Mozile.prototype.executeCommand = function(id, event) {
	var f = new Array();
	f["File"] = "core/interface.js";
	f["Function"] = "Mozile.executeCommand()";
	this.debug(f,1,"Executing command "+ id +" "+ event);

	// Remove the "-Button" or "-Menuitem" part of the id, if there is one.
	var cleanId = /(.*)(\-Button|\-Menuitem)$/;
	var result = cleanId.exec(id);
	//alert(result);
	var commandId;
	if(result) commandId = result[1];
	else commandId = id;

	if(!this.commandIsRegistered(commandId)) return false;
	// If the command id is registered, call the command method of the corresponding command object.
	else {
		var command = this.commandList[commandId];
		
		// If the keyCounter is not 0, then something has changed since the last command, so store the state.
		if(this.keyCounter != 0) this.storeState(command);
		
		// Execute the command
		result = command.command(event);
		
		// If there is some result, and this is not an undo or redo command, store the state after the change.
		if(result && command.id!="Mozile-Undo" && command.id!="Mozile-Redo") this.storeState(command);

		this.updateToolbar(true);
		return true;
	}

}








/****  Mozile Command Object ****/

/** Mozile Command Object -
 * Creates a Mozile command object, which controls the button, menuitem, and code associated with a Mozile editing command.
 * <p>Configuration String format (particular command objects may have other options): "MozileCommandObject: id=CommandId, label='Command Label', tooltip='Command Tool Tip', image='/path/to/image.file', accesskey=C, accelerator='Command-C', buttonPosition=0, menuPosition=0, debugLevel=0"
 * <p>Accelerator format must follow this sequence: "Command-Meta-Control-Alt-Shift-Key". Mozile will check the UserAgent string for the browser, and replace "Command" with "Control" on Linux and Windows, or "Meta" on Macintosh. One of "Command", "Control" or "Meta" must be present, and "Key" is a single uppercase character.
 * @constructor
 *
 * @param configString A configuration string, following the above format.
 * @return The newly created command object.
 */
function MozileCommand() {

	
	/**** DEFINITIONS ****/
	// Define the properties for this object.

	/** Mozile Command - Type -
	 * Denotes this object as a command.
	 */
	this.type = "MozileCommand";

	/** Mozile Command - Id -
	 * The unique id by which this command is known. The usual format is "Mozile-ModuleName-CommandName".
	 */
	this.id = null;

	/** Mozile Command - Label -
	 * A word or two describing this command.
	 */
	this.label = null;

	/** Mozile Command - Tooltip -
	 * A short phrase describing this command.
	 */
	this.tooltip = null;

	/** Mozile Command - Image -
	 * The path to the image associated with this command, if any.
	 */
	this.image = null;

	/** Mozile Command - Access Key -
	 * The keyboard access key for the menuitem, or "null" for none.
	 */
	this.accesskey = null;

	/** Mozile Command - Accelerator -
	 * The keyboard shortcut for this command, or "null" for none.
	 */
	this.accelerator = null;
	
	/** Mozile Command - Namespace -
	 * The XUL namespace in which elements will be created.
	 */
	this.namespace = XULNS;
	
	/** Mozile Command - Debug Level -
	 * An integer indicating how verbose debugging should be. 0 means only critical errors are shown. Higher values mean more verbose debugging: 1="very important", 2="important", 3="normal", 4="not important"
	 */
	this.debugLevel = 0;

	/** Mozile Command - Button -
	 * The button element this command, or "null" for none.
	 */
	this.button = null;

	/** Mozile Command - Menuitem -
	 * The menuitem for this command, or "null" for none.
	 */
	this.menuitem = null;
	

	if(arguments.length > 0){
		var configArray = arguments[0];
		this.init(configArray);
	}
	else return true;

	return "Success";
	
}




/** Mozile Command - Init  -
 * Initializes the Mozile Command object, setting various properties based on the configArray.
 *
 * @param configArray An array of configuration opions, as created by Mozile.parseConfig().
 * @return True if successful, an error message otherwise.
 */
MozileCommand.prototype.init = function(configArray) {
	
	/**** VALIDATION ****/
	// Check to make sure that values are valid, then set them.
	
	this.type = configArray['name'];
	
	// Check to see if a valid id was provided.
	if(configArray['id'] && typeof(configArray['id'])!="null") {
		this.id = configArray['id'];
	}
	else {
		return "Error initializing MozileCommandList object -- invalid id provided: "+configArray['id'];
	}
	
	// Check to see if a valid label was provided.
	if(configArray['label'] && typeof(configArray['label'])!="null") {
		this.label = configArray['label'];
	}
	else {
		return "Error initializing MozileCommandList object -- invalid label provided: "+configArray['label'];
	}
	
	// Check to see if a valid tooltip was provided.
	if(configArray['tooltip']) {
		this.tooltip = configArray['tooltip'];
	}
	
	// Check to see if a valid image was provided.
	if(configArray['image']) {
		this.image = configArray['image'];
	}
	
	// Check to see if a valid accesskey was provided.
	if(configArray['accesskey']) {
		this.accesskey = configArray['accesskey'];
	}
	
	// Check to see if a valid accelerator was provided.
	if(configArray['accelerator']) {
		this.accelerator = configArray['accelerator'];
	}
	
	// Check to see if debug is an integer
	if(configArray['debugLevel']) {
		if(typeof(configArray['debugLevel'])=="number") {
			this.debugLevel = configArray['debugLevel'];
		}
		else {
			this.debugLevel = mozile.debugLevel;
		}
	}
	
	return true;
		
}


// Add the mozileDebug function as a method of the MozileCommand object.
MozileCommand.prototype.debug = mozileDebug;





/** Mozile Command - Create Button -
 * Creates an XUL button element for this command.
 * 
 * @return Button element.
 */
MozileCommand.prototype.createButton = function() {
	var f = new Array();
	f["File"] = "core/interface.js";
	f["Function"] = "MozileCommand.createButton()";
	this.debug(f,1,"Creating button");

	var button = document.createElementNS(XULNS, "toolbarbutton");
	button.setAttribute("id", this.id+"-Button");
	button.setAttribute("class", "mozileButton");
	button.setAttribute("image", this.image);
	button.setAttribute("label", this.label);
	if(this.tooltip) button.setAttribute("tooltiptext", this.tooltip);
	
	this.button = button;
	return button;
}




/** Mozile Command - Create Menuitem -
 * Creates an XUL menuitem element for this command.
 * 
 * @return The new menuitem element.
 */
MozileCommand.prototype.createMenuitem = function() {
	var f = new Array();
	f["File"] = "core/interface.js";
	f["Function"] = "MozileCommand.createMenuitem()";
	this.debug(f,1,"Creating menuitem");
	
	var menuitem = document.createElementNS(XULNS, "menuitem");
	menuitem.setAttribute("id", this.id+"-Menuitem");
	menuitem.setAttribute("class", "mozileMenuitem");
	//menuitem.setAttribute("type", "checkbox");
	menuitem.setAttribute("label", this.label);
	if(this.tooltip) menuitem.setAttribute("tooltiptext", this.tooltip);
	if(this.accesskey) menuitem.setAttribute("accesskey", this.accesskey);
	
	this.menuitem = menuitem;
	return menuitem;

}


/** Mozile Command - Is Active -
 * Checks to see if the command is active in the current selection. In the general case, isActive always returns false. But other Mozile command objects define it in non-trivial ways.
 * 
 * @return Always false.
 */
MozileCommand.prototype.isActive = function() {
	var f = new Array();
	f["File"] = "core/interface.js";
	f["Function"] = "MozileCommand.isActive()";
	this.debug(f,0,"Checking to see if this command is active");
	
	return false;
}

/** Mozile Command - Update Button -
 * Updates the XUL button and menu item for this command, setting its "active" or "checked" attribute to true if this.isActive() returns true, and to "false" otherwise.
 * 
 * @return True if the button and menuitem are active, and false otherwise 
 */
MozileCommand.prototype.update = function() {
	var f = new Array();
	f["File"] = "core/interface.js";
	f["Function"] = "MozileCommand.update()";
	this.debug(f,0,"Updating button and menuitem");
	
	var button = this.button;
	var menuitem = this.menuitem;
	
	// If isActive returns true, set the attributes to true.
	if(this.isActive()) {
		if(button) {
			button.setAttribute("active",true);
		}
		if(menuitem) {
			menuitem.setAttribute("checked",true);
		}
		return true;
	}
	// If isActive returns false, set the attributes to false.
	else {
		if(button) {
			button.setAttribute("active",false);
		}
		if(menuitem) {
			menuitem.setAttribute("checked",false);
		}
		return false;
	}

}




/** Mozile Command - Command -
 * Implements the editing command for this object. For the general Mozile command object, the command merely display an alert, but commands derived from this will implement their own code.
 * 
 * @param event The event which triggered this command.
 * @return True if successful.
 */
MozileCommand.prototype.command = function(event) {
	var f = new Array();
	f["File"] = "core/interface.js";
	f["Function"] = "MozileCommand.command()";
	this.debug(f,1,"Executing command "+event);

	alert("Command! "+ this.id +" "+ event);
	return true;

}







/** Mozile Command List Object -
 * Creates a Mozile command list object, inheriting from MozileCommand, which controls the box and menu associated with a group of Mozile editing commands.
 * @constructor
 *
 * @param configArray An array of configuration information and options, created from a congiuration string by the Mozile.parseConfig() method.
 * @return A string indicating success or an error.
 */
MozileCommandList.prototype = new MozileCommand();
MozileCommandList.prototype.constructor = MozileCommandList;
MozileCommandList.superclass = MozileCommand.prototype;

function MozileCommandList() {

	if(arguments.length > 0){
		var configArray = arguments[0];
		this.init(configArray);
	}
	else return true;


	/** Mozile Command List - Button -
	 * The box element for this command, or "null" for none. The box contains the buttons for the commands in this list.
	 */
	this.box = null;
	
	/** Mozile Command List - Menu -
	 * The menu element for this command, or "null" for none.
	 */
	this.menu = null;
	
	/** Mozile Command List - Command List -
	 * An array of all the command (list) objects in this list. Key=id, value=command object.
	 */
	this.commandList = new Array();
	
	/** Mozile Command List - Button List -
	 * An array of all the command (list) objects which use buttons in this list, in the order in which they appear.
	 */
	this.buttonList = new Array();
	
	/** Mozile Command List - Menu List -
	 * An array of all the command (list) objects which use menus or menuitems in this list, in the order in which they appear.
	 */
	this.menuList = new Array();

	return true;
}



/** Mozile Command List - Create Command -
 * Creates a new command object from a configuration string, registers it with the "mozile" object, and attaches it to this MozileCommandList.
 * 
 * Configuration String format: "MozileCommandObject: id=CommandId, label='Command Label', tooltip='Command Tool Tip', image='/path/to/image.file', accesskey=C, accelerator='Meta-C', buttonPosition=0, menuPosition=0"
 * The type=MozielCommandObject
 * Executes string "command = new MozileCommandObject(optionsArray)"
 *
 * @param configString A configuration string which describes the command to be created.
 * @return The new command object.
 */
MozileCommandList.prototype.createCommand = function(configString) {
	var f = new Array();
	f["File"] = "core/interface.js";
	f["Function"] = "MozileCommandList.createCommand()";
	this.debug(f,1,"Creating command "+ configString);

	// Parse the configString.
	var configArray = mozile.parseConfig(configString);
	
	var name = configArray['name'];
	var command;
	
	// Define the command creation string. I.e. "command = new MozileCommand(configArray)"
	var create = "command = new "+name+"(configArray)";
	
	// Evaluate the command creation string.
	eval(create);
	
	// If a buttonPosition has been specified, set the buttonPosition variable.
	var buttonPosition = null;
	if(configArray['buttonPosition'] != null) buttonPosition = configArray['buttonPosition'];
	
	// If a menuPosition has been specified, set the menuPosition variable.
	var menuPosition = null;
	if(configArray['menuPosition'] != null) menuPosition = configArray['menuPosition'];
	
	// Register this command.
	this.registerCommand(command, buttonPosition, menuPosition);
	
	return command;

}




/** Mozile Command List - Register Command -
 * Registers a command with this command list object and the "mozile" object.
 * 
 * @param commadn The command object to be registered.
 * @param buttonPosition An integer indicating the index in the buttonList where this command's button should be added. If none is given, it is added to the end of the list.
 * @param menuPosition An integer indicating the index in the menuList where this command's menuitem should be added. If none is given, it is added to the end of the list.
 * @return Always true.
 */
MozileCommandList.prototype.registerCommand = function(command, buttonPosition, menuPosition) {
	var f = new Array();
	f["File"] = "core/interface.js";
	f["Function"] = "MozileCommandList.registerCommand()";
	this.debug(f,1,"Registering command "+ command +" with id "+ command.id +" at "+ buttonPosition +" and "+ menuPosition);

	// Register the command with the global mozile object instance.
	var id = command.id
	if(command.type!="MozileCommandList") {
		mozile.registerCommand(command);
	}
	
	// Add the command to the local commandList.
	this.commandList[id] = command;
	
	// If no button position is given, add this command to the end of the buttonList.
	if(buttonPosition == null) {
		this.buttonList.push(command);
	}
	else {
		// if there is no other button at that index, insert this one.
		if(this.buttonList[buttonPosition]==null) {
			this.buttonList[buttonPosition] = command;
		}
		// if there is already a button there, splice this one in before the old one.
		else {
			this.buttonList.splice(buttonPosition, 0, command);
		}
	}
	
	// If a menu position is given, add this command to the menuList.
	if(menuPosition == null) {
		this.menuList.push(command);
	}
	else {
		// if there is no other menu item at that index, insert this one.
		if(this.menuList[menuPosition]==null) {
			this.menuList[menuPosition] = command;
		}
		// if there is already a menu item there, splice this one in before the old one.
		else {
			this.menuList.splice(menuPosition, 0, command);
		}
	}
	
	return true;
}



/** Mozile Command List - Unregister Command -
 * Removes a command from the local and global commandLists, as well as the buttonList and menuList.
 * 
 * @param id The id of the command to be removed.
 * @return Always true.
 */
MozileCommandList.prototype.unregisterCommand = function(id) {
	var f = new Array();
	f["File"] = "core/interface.js";
	f["Function"] = "MozileCommandList.unregisterCommand()";
	this.debug(f,1,"Unregistering command "+ id);

	// Unregister globally.
	mozile.unregisterCommand(command);
	
	// Unregister locally.
	this.commandList[id]=null;
	
	// Remove from buttonList.
	var i;
	for(i=0; i < this.buttonList; i++) {
		if(this.buttonList[i].id == id) {
			this.buttonList.splice(i,0);
			i--;
		}
	}
	
	// Remove from menuList.
	for(i=0; i < this.menuList; i++) {
		if(this.menuList[i].id == id) {
			this.menuList.splice(i,0);
			i--;
		}
	}

	return true;
}




/** Mozile Command List - Create Box -
 * Creates an XUL box to hold the buttons fro this command list.
 * 
 * @return The XUL box element.
 */
MozileCommandList.prototype.createBox = function() {
	var f = new Array();
	f["File"] = "core/interface.js";
	f["Function"] = "MozileCommandList.createBox()";
	this.debug(f,1,"Creating box");

	// Create the box element
	var box = document.createElementNS(XULNS, "box");
	box.setAttribute("id", this.id+"-Box");
	box.setAttribute("class", "mozileBox");
	//box.setAttribute("image", this.image);
	box.setAttribute("label", this.label);
	if(this.tooltip) box.setAttribute("tooltiptext", this.tooltip);

	// Add all the buttons from the buttonList, making sure that they exist first.
	var button;
	for(var i=0; i < this.buttonList.length; i++) {
		if(this.buttonList[i]) {
			button = this.buttonList[i].button;
			if(!button || button=="") {
				this.buttonList[i].createButton();
				button = this.buttonList[i].button;
			}
			box.appendChild(button);
		}
	}
	
	this.box = box;
	return box;
}



/** Mozile Command List - Create Button -
 * Creates an XUL button for this command list.
 * 
 * @return The XUL button element.
 */
MozileCommandList.prototype.createButton = function() {
	var f = new Array();
	f["File"] = "core/interface.js";
	f["Function"] = "MozileCommandList.createButton()";
	this.debug(f,1,"Creating button");

	// Create the button element.
	var button = document.createElementNS(XULNS, "toolbarbutton");
	button.setAttribute("id", this.id+"-Button");
	button.setAttribute("class", "mozileButton");
	button.setAttribute("image", this.image);
	button.setAttribute("label", this.label);
	if(this.tooltip) button.setAttribute("tooltiptext", this.tooltip);
	button.setAttribute("type", "menu");
	
	// Create the menupopup child element.
	var menuPopup = document.createElementNS(XULNS, "menupopup");
	button.appendChild(menuPopup);

	// Add every menuitem on the menuList to the menupopup, making sure they exist first.
	var menuitem,menu;
	for(var i=0; i < this.menuList.length; i++) {
		if(this.menuList[i]) {
			if(this.menuList[i].type!="MozileCommandList") {
				menuitem = this.menuList[i].menuitem;
				if(!menuitem || menuitem=="") {
					this.menuList[i].createMenuitem();
					menuitem = this.menuList[i].menuitem;
				}
				menuPopup.appendChild(menuitem);
			}
			// if it's a command list
			else {
				menu = this.menuList[i].menu;
				if(!menu || menu=="") {
					this.menuList[i].createMenu();
					menu = this.menuList[i].menu;
				}
				menuPopup.appendChild(menu);
			}
		}
	}
	
	this.button = button;
	return button;
}




/** Mozile Command List - Create Menu -
 * Creates an XUL menu element for this command.
 * 
 * @return The XUL menu element.
 */
MozileCommandList.prototype.createMenu = function() {
	var f = new Array();
	f["File"] = "core/interface.js";
	f["Function"] = "MozileCommandList.createMenu()";
	this.debug(f,1,"Creating menu");
	
	// Create the menu element
	var menu = document.createElementNS(XULNS, "menu");
	menu.setAttribute("id", this.id+"-Menu");
	menu.setAttribute("class", "mozileMenu");
	menu.setAttribute("label", this.label);
	if(this.tooltip) menu.setAttribute("tooltiptext", this.tooltip);
	if(this.accesskey) menu.setAttribute("accesskey", this.accesskey);
	
	// Create the menupopup child element.
	var menuPopup = document.createElementNS(XULNS, "menupopup");
	menu.appendChild(menuPopup);

	// For each menuitem on the menuList, add it to the menupopup (making sure it exists first).
	var menuitem;
	for(var i=0; i < this.menuList.length; i++) {
		if(this.menuList[i]) {
			menuitem = this.menuList[i].menuitem;
			if(!menuitem || menuitem=="") {
				this.menuList[i].createMenuitem();
				menuitem = this.menuList[i].menuitem;
			}
			menuPopup.appendChild(menuitem);
		}
	}
	
	this.menu = menu;
	return menu;

}





/** Mozile Command List - Update Box -
 * Updates the state of all the buttons for this command.
 * TODO: Not currently used, so it might not be useful at all.
 * 
 * @return Always true.
 */
MozileCommandList.prototype.updateBox = function() {
	var f = new Array();
	f["File"] = "core/interface.js";
	f["Function"] = "MozileCommandList.updateBox()";
	this.debug(f,1,"Updating box");

	return true;

}




/** Mozile Command List - Update Menu -
 * Updates the state of the menu for this command.
 * TODO: Not currently used, so it might not be useful at all.
 * 
 * @return Always true.
 */
MozileCommandList.prototype.updateMenu = function() {
	var f = new Array();
	f["File"] = "core/interface.js";
	f["Function"] = "MozileCommandList.updateMenu()";
	this.debug(f,1,"Updating menu");

	return true;

}







/**** Mozile User Interface "Dialogs" ****/


/** Mozile - Show About Interface -
 * Shows the About interface, by setting "collapsed" to false, and setting height and width values. The About interface displays the version, URL, and developers of Mozile.
 * 
 * @return Always true.
 */
Mozile.prototype.showAboutInterface = function() {
	this.aboutInterface.collapsed = false;
	
	this.aboutInterface.style.width = "450px";
	this.aboutInterface.style.height = "300px";
	
	mozileCenterNode(this.aboutInterface);
	
}

/** Mozile - Hide About Interface -
 * Shows the About interface, by setting "collapsed" to true, and setting height and width values to zero.
 * 
 * @return Always true.
 */
Mozile.prototype.hideAboutInterface = function() {
	this.aboutInterface.collapsed = true;
	
	this.aboutInterface.style.width = "0px";
	this.aboutInterface.style.height = "0px";
	
	mozileDisappearNode(this.aboutInterface);
	
}



/** Mozile - Show Save Interface -
 * Shows the Save interface, by setting "collapsed" to false, and setting height and width values. The Save interface presents the user with Save As options.
 * 
 * @return Always true.
 */
Mozile.prototype.showSaveInterface = function() {
	this.saveInterface.collapsed = false;
	
	this.saveInterface.style.width = "355px";
	this.saveInterface.style.height = "270px";
	
	mozileCenterNode(this.saveInterface);
	
}

/** Mozile - Hide Save Interface -
 * Shows the Save interface, by setting "collapsed" to true, and setting height and width values to zero.
 * 
 * @return Always true.
 */
Mozile.prototype.hideSaveInterface = function() {
	this.saveInterface.collapsed = true;
	
	this.saveInterface.style.width = "0px";
	this.saveInterface.style.height = "0px";
	
	mozileDisappearNode(this.saveInterface);
	
}


/** Mozile - Show Source Interface -
 * Shows the Source interface, by setting "collapsed" to false, and setting position values. The Source interface displays the source code of the document.
 * 
 * @return Always true.
 */
Mozile.prototype.showSourceInterface = function() {
	this.sourceInterface.collapsed = false;
	
	//this.sourceInterface.style.width = window.innerWidth - 50 +"px";
	//this.sourceInterface.style.height = window.innerHeight - 50 +"px";
	var space = "50px";
	this.sourceInterface.style.top = space;
	this.sourceInterface.style.left = space;
	this.sourceInterface.style.right = space;
	this.sourceInterface.style.bottom = space;
	
	//mozileCenterNode(this.sourceInterface);
	
}

/** Mozile - Hide Source Interface -
 * Shows the Source interface, by setting "collapsed" to true, and setting height and width values to zero.
 * 
 * @return Always true.
 */
Mozile.prototype.hideSourceInterface = function() {
	this.sourceInterface.collapsed = true;
	
	this.sourceInterface.style.top = "-500px";
	this.sourceInterface.style.left = "-500px";
	this.sourceInterface.style.right = "-500px";
	this.sourceInterface.style.bottom = "-500px";
	
	mozileDisappearNode(this.sourceInterface);

}



/** Mozile - Show Message Interface -
 * Shows the Message interface, by setting "collapsed" to false, and setting height and width values. The Message interface displays debugging and status messages.
 * 
 * @return Always true.
 */
Mozile.prototype.showMessageInterface = function() {
	this.messageInterface.collapsed = false;
	
	this.messageInterface.style.width = window.innerWidth - 50 +"px";
	this.messageInterface.style.height = window.innerHeight - 50 +"px";

	var bugs = document.getAnonymousElementByAttribute(mozile.saveInterface.parentNode, "id", "bugs");
	bugs.style.width = window.innerWidth - 60 +"px";
	bugs.style.height = window.innerHeight - 150 +"px";
	
	mozileCenterNode(this.messageInterface);
	
}

/** Mozile - Hide Message Interface -
 * Shows the Message interface, by setting "collapsed" to true, and setting height and width values to zero.
 * 
 * @return Always true.
 */
Mozile.prototype.hideMessageInterface = function() {
	this.messageInterface.collapsed = true;
	
	this.messageInterface.style.width = "0px";
	this.messageInterface.style.height = "0px";
	
	mozileDisappearNode(this.messageInterface);
	
}




/** Mozile Center Node -
 * Sets the CSS "left" and "top" properties such that the node is centered inside the current window.
 * 
 * @return Always true.
 */
function mozileCenterNode(node) {
	var left = (window.innerWidth - node.boxObject.width) / window.innerWidth * 50 ;
	node.style.left = left + "%";
	var top = (window.innerHeight - node.boxObject.height) / window.innerHeight * 50;
	node.style.top = top + "%";
}




/** Mozile Disappear Node -
 * Sets the CSS "left" and "top" properties such that the node is rendered far from the edge of the window.
 * 
 * @return Always true.
 */
function mozileDisappearNode(node) {
	node.style.left = "-500px";
	node.style.top = "-500px";
}








/**** Mozile Save Interface Functions ****/

// Mozile Save Interface Globals
var mozileSavePreset;
var mozileCustomPreset;
var mozileSaveContent;
var mozileSaveFormat;
var mozileSaveMethod;



/** Mozile Save Init -
 * Initialize the Save interface. First, find the elements for the global variables, and then set them up.
 * 
 * @return Always true.
 */	
function mozileSaveInit() {
	mozileSavePreset = document.getAnonymousElementByAttribute(mozile.saveInterface.parentNode, "id", "mozileSavePresetList");
	mozileCustomPreset = document.getAnonymousElementByAttribute(mozile.saveInterface.parentNode, "id", "mozileCustomPresetItem");
	mozileSaveContent = document.getAnonymousElementByAttribute(mozile.saveInterface.parentNode, "id", "mozileSaveContentList");
	mozileSaveFormat = document.getAnonymousElementByAttribute(mozile.saveInterface.parentNode, "id", "mozileSaveFormatList");
	mozileSaveMethod = document.getAnonymousElementByAttribute(mozile.saveInterface.parentNode, "id", "mozileSaveMethodList");
	
	
	// Clear save method menuitems
	var child = mozileSaveMethod.firstChild.firstChild;
	var nextChild;
	while(child) {
		nextChild = child.nextSibling;
		mozileSaveMethod.firstChild.removeChild(child);
		child = nextChild;
	}
	
	// Generate save method menuitems
	var menuitem;
	for(key in mozile.saveList) {
		if(key != "default" && key != "custom") {
			menuitem = document.createElementNS(XULNS, "menuitem");
			menuitem.setAttribute("value", mozile.saveList[key]["value"]);
			menuitem.setAttribute("label", mozile.saveList[key]["label"]);
			menuitem.setAttribute("oncommand", "mozileSaveChanged()");
			mozileSaveMethod.firstChild.appendChild(menuitem);
		}
	}
	

	// Apply custom settings if they exist
	if(mozile.saveList["custom"]) {
		mozileSavePreset.selectedItem = mozileCustomPreset;
		mozileCustomPreset.setAttribute("disabled","false");
	
		// Set save content 
		if(mozile.saveList["custom"]["content"]) {
			mozileSaveContent.value = mozile.saveList["custom"]["content"];
		}
	
		// Set save format 
		if(mozile.saveList["custom"]["format"]) {
			mozileSaveFormat.value = mozile.saveList["custom"]["format"];
		}
	
		// Set save method 
		if(mozile.saveList["custom"]["value"]) {
			mozileSaveMethod.value = mozile.saveList["custom"]["value"];
		}

		// Try to set the URL field		
		if(mozile.saveList["custom"]["url"] != null) {
			document.getAnonymousElementByAttribute(mozile.saveInterface.parentNode, "id", "mozileSaveToURL").value = mozile.saveList["custom"]["url"];
		}
	}

	// Otherwise, apply default settings
	else {
	
		// Apply default settings, if they exist.
		if(mozile.saveList["default"]) {
			mozileSaveMethod.value = mozile.saveList["default"]["value"];
		
			// Try to set the URL field
			if(mozile.saveList["default"]["url"] != null) {
				document.getAnonymousElementByAttribute(mozile.saveInterface.parentNode, "id", "mozileSaveToURL").value = mozile.saveList["default"]["url"];
			}
		}
		
		// If neither custom nor default settings exist, us this
		else {
			mozileSaveMethod.selectedIndex = 0;
		}
	
	}

	mozileMethodFields();
}


/** Mozile Method Fields -
 * Show and hide various fields. Right now we only have one: mozileSaveURLBox.
 * 
 * @return Always true.
 */	
function mozileMethodFields() {
	var f = new Array();
	f["File"] = "core/interface.js";
	f["Function"] = "mozileMethodFields()";
	mozile.debug(f,1,"Setting Method Fields");
	
	var method = mozileSaveMethod.value;
	
	// var mozileFileBox = document.getAnonymousElementByAttribute(mozile.saveInterface.parentNode, "id", "mozileSaveFileBox");
	var mozileURLBox = document.getAnonymousElementByAttribute(mozile.saveInterface.parentNode, "id", "mozileSaveURLBox");
	
	// mozileFileBox.collapsed = true;
	mozileURLBox.collapsed = true;
			
	// if(mozile.saveList[method]["file"] != null) {
	// 	mozileFileBox.collapsed = false;
	// }
	//alert(method +" "+ mozile.saveList[method]["showURL"]);
	if(mozile.saveList[method]["showURL"] == true) {
		mozileURLBox.collapsed = false;
	}
	
	mozile.debug(f,3, method +" "+ mozileURLBox +" "+ mozile.saveList[method]["showURL"]);
}

/** Mozile Restore Default -
 * Restore the page default settings, if they exist
 * 
 * @return Always true.
 */	
function mozileRestoreDefault() {
	if(mozile.saveList["default"]) {
		mozileSaveContent.value = mozile.saveList["default"]["content"];
		mozileSaveFormat.value = mozile.saveList["default"]["format"];
		mozileSaveMethod.value = mozile.saveList["default"]["value"];
		if(mozile.saveList["default"]["url"]) {
			document.getAnonymousElementByAttribute(mozile.saveInterface.parentNode, "id", "mozileSaveToURL").value = mozile.saveList["default"]["url"];
		}
	}
}


/** Mozile Restore Custom -
 * Restore the custom settings, if they exist. Only works within dialog sessions, not between them.
 * 
 * @return Always true.
 */	
function mozileRestoreCustom() {
	if(mozile.saveList["default"]) {
		mozileSaveContent.value = mozile.saveList["custom"]["content"];
		mozileSaveFormat.value = mozile.saveList["custom"]["format"];
		mozileSaveMethod.value = mozile.saveList["custom"]["value"];
		if(mozile.saveList["custom"]["url"]) {
			document.getAnonymousElementByAttribute(mozile.saveInterface.parentNode, "id", "mozileSaveToURL").value = mozile.saveList["custom"]["url"];
		}
	}
}


/** Mozile Save Changes -
 * If there have been changes, set the "preset" menu to "custom" and store the changes.
 * 
 * @return Always true.
 */	
function mozileSaveChanged() {
	mozileMethodFields();
	mozileStoreCustom();
	mozileCustomPreset.setAttribute("disabled","false");
	mozileSavePreset.selectedItem = mozileCustomPreset;
	
}


/** Mozile Save Changes -
 * When the "save" button is pressed, collect all the values that have been set,  and call Mozile.save().
 * 
 * @return Always true.
 */	
function mozileSaveDone() {

	if(mozileSavePreset.value == "page") {
		mozile.saveList["custom"] = null;
		var method = mozileSaveMethod.value
		mozile.saveList[method]["content"] = mozileSaveContent.value;
		mozile.saveList[method]["format"] = mozileSaveFormat.value;
		mozile.saveList[method]["method"] = mozileSaveMethod.value;
	}
	else {
		mozileStoreCustom();
	}
	//alert("Saving");
	
	mozile.saveFlag = true;
	
	mozile.save();
	
	mozile.hideSaveInterface();
}


/** Mozile Store Custom -
 * Store the custom values of all the menus and fields.
 * 
 * @return Always true.
 */	
function mozileStoreCustom() {
	mozile.saveList["custom"] = new Array();
	mozile.saveList["custom"]["value"] = mozile.saveList[mozileSaveMethod.value]["value"];
	mozile.saveList["custom"]["label"] = mozile.saveList[mozileSaveMethod.value]["label"];
	mozile.saveList["custom"]["function"] = mozile.saveList[mozileSaveMethod.value]["function"];
	mozile.saveList["custom"]["content"] = mozileSaveContent.value;
	mozile.saveList["custom"]["format"] = mozileSaveFormat.value;
	mozile.saveList["custom"]["method"] = mozileSaveMethod.value;
	if(!document.getAnonymousElementByAttribute(mozile.saveInterface.parentNode, "id", "mozileSaveURLBox").collapsed) {
		mozile.saveList["custom"]["url"] = document.getAnonymousElementByAttribute(mozile.saveInterface.parentNode, "id", "mozileSaveToURL").value;
	}
}




/**** Mozile Debug Interface Functions ****/

	
/** Mozile Debug Init -
 * Initializes the debugging dialog.
 *
 * @param event The event object which triggers the initialization.
 * @return Always true.
 */
function mozileDebugInit() {
	//alert(opener +" "+ opener.mozileTarget);
	//if (event.target != document) return;
	//document.getElementById('version').value=opener.mozileTarget.mozile.version;
	mozileDebugReset();
}

	
/** Mozile Debug Reset -
 * Clears the filter and rebuilds the bug list.
 *
 * @param event The event object which triggers the initialization.
 * @return Always true.
 */
function mozileDebugReset() {
	//var mozileDebugList = mozileDebugList;
	document.getAnonymousElementByAttribute(mozile.sourceInterface.parentNode,'id','filterText').value = "";
	document.getAnonymousElementByAttribute(mozile.sourceInterface.parentNode,'id','filterLevel').selectedIndex=0;
	mozileDebugFilter();
}

	
/** Mozile Debug Clear -
 * Clears the list and empties the mozileDebugArray.
 *
 * @param event The event object which triggers the initialization.
 * @return Always true.
 */
function mozileDebugClear() {
	mozileDebugList = new Array();
	mozileDebugReset();
}

	
/** Mozile Debug Filter -
 * Rebuilds the bug list, filtering based on the content of the filterText input field.
 *
 * @return Always true.
 */
function mozileDebugFilter() {
	//alert("Filtering Level: "+node.value);	
	
	var selected = document.getAnonymousElementByAttribute(mozile.sourceInterface.parentNode,'id','filterLevel').value;		// get the value of the Filter Level menu
	var text = document.getAnonymousElementByAttribute(mozile.sourceInterface.parentNode,'id','filterText').value;		// get the Filter Text
	
	//var mozileDebugList = mozileDebugList;
	var vbox = document.getAnonymousElementByAttribute(mozile.sourceInterface.parentNode,'id',"bugs");
	var count = 0;
	mozileDebugRemoveChildren(vbox);
	
	// Loop over all the bug entries
	for(var i=0; i < mozileDebugList.length; i++) {
		var bug = mozileDebugList[i];

			// If the selected value is "status", and this entry is not a "Status Message", then ignore it.
		if(selected == "status" && !bug[1]["Status Message"] ) continue;
			// if the selected level is lower, ignore this entry
		if(selected > bug[2]) continue;			
			// search each of the fields, and keep a total
		var search = bug[0].search(text) + bug[1]["File"].search(text) + bug[1]["Function"].search(text) + bug[3].search(text);
			// if all the results were "-1", then skip this entry
		if( search == -4 ) continue;
		
		count++;
		
		vbox.appendChild(mozileDebugEntry(mozileDebugList[i]));
	}
	
	document.getAnonymousElementByAttribute(mozile.sourceInterface.parentNode,'id','messageCount').value="Showing "+count+" of "+mozileDebugList.length+" Messages";

}

	
/** Mozile Debug Entry -
 * Create an entry for the debugging item, which consists of an hbox filled with description elements.
 *
 * @param bug The array for this bug entry.
 * @return The empty node.
 */
function mozileDebugEntry(bug) {
	var hbox = document.createElementNS(XULNS, "hbox");
	hbox.setAttribute("class", "level"+bug[2]);
	hbox.setAttribute("align", "center");
	
	var datestamp = document.createElementNS(XULNS, "description");
	datestamp.setAttribute("class","datestamp");
	datestamp.setAttribute("value", bug[0]);
	hbox.appendChild(datestamp);
	
	var button = document.createElementNS(XULNS, "button");
	button.setAttribute("class","functionButton");
	button.setAttribute("type", "menu");
	button.setAttribute("image", mozile.root+"images/info.png");
	
	var menupopup = document.createElementNS(XULNS, "menupopup");

	var functions;
	for(key in bug[1]) {
		functions = document.createElementNS(XULNS, "description");
		functions.setAttribute("class","functions");
		functions.appendChild(document.createTextNode(key +" = "+ bug[1][key]));
		menupopup.appendChild(functions);
	}
	functions = document.createElementNS(XULNS, "description");
	functions.setAttribute("class","functions");
	functions.appendChild(document.createTextNode("Level = "+ bug[2]));
	menupopup.appendChild(functions);
	
	button.appendChild(menupopup);
	hbox.appendChild(button);
	
	var message = document.createElementNS(XULNS, "description");
	message.setAttribute("class","message");
	message.appendChild(document.createTextNode(bug[3]));
	hbox.appendChild(message);

	return hbox;
}

/** Mozile Array String -
 * Outputs the keys and values of an array as a string.
 *
 * @param arr Array. The array to be rendered to a string.
 * @return The new string.
 */
function mozileArrayString(arr) {
	var result = new Array();
	for(key in arr) {
		result.push(key +"="+ arr[key]);
	}
	return result.join("\n");
}

	
/** Mozile Debug Remove Children -
 * Removes all children of the given node.
 *
 * @param node The node to be cleaned up.
 * @return The empty node.
 */
function mozileDebugRemoveChildren(node) {	
	while(node.childNodes.length) {
		node.removeChild(node.firstChild);
	}
	return node;
}




/**** Activate Configuration ****/
// When everything is ready, call the mozileConfiguration function defined in mozile.js
// This will start loading the modules into the document.
try {
	mozileConfiguration();
}
catch(e) {
	alert("Error in MozileCore.js when calling mozileConfiguration: "+e);
}

