/* ***** BEGIN LICENSE BLOCK *****
 * Version: MPL 1.1/GPL 2.0/LGPL 2.1
 *
 * The contents of this file are subject to the Mozilla Public License Version
 * 1.1 (the "License"); you may not use this file except in compliance with
 * the License. You may obtain a copy of the License at
 * http://www.mozilla.org/MPL/
 *
 * Software distributed under the License is distributed on an "AS IS" basis,
 * WITHOUT WARRANTY OF ANY KIND, either express or implied. See the License
 * for the specific language governing rights and limitations under the
 * License.
 *
 * The Original Code is hultmann localization tools.
 *
 * The Initial Developer of the Original Code is
 * Jeferson Hultmann <hultmann@gmail.com>
 * Portions created by the Initial Developer are Copyright (C) 2006
 * the Initial Developer. All Rights Reserved.
 *
 * Contributor(s):
 *
 * Alternatively, the contents of this file may be used under the terms of
 * either of the GNU General Public License Version 2 or later (the "GPL"),
 * or the GNU Lesser General Public License Version 2.1 or later (the "LGPL"),
 * in which case the provisions of the GPL or the LGPL are applicable instead
 * of those above. If you wish to allow use of your version of this file only
 * under the terms of either the GPL or the LGPL, and not to allow others to
 * use your version of this file under the terms of the MPL, indicate your
 * decision by deleting the provisions above and replace them with the notice
 * and other provisions required by the GPL or the LGPL. If you do not delete
 * the provisions above, a recipient may use your version of this file under
 * the terms of any one of the MPL, the GPL or the LGPL.
 *
 * ***** END LICENSE BLOCK ***** */


namespace Hultmann.Zip
{


internal sealed class ZipOutputStream : System.IO.Stream
{

private readonly System.IO.Stream                                m_theArchive;
private          System.Collections.Generic.List<ZipEntryWriter> m_compressedEntries = new System.Collections.Generic.List<ZipEntryWriter>();
private          bool                                            m_finished = false;
private          ZipEntry                                        m_infoEntry = null;
private          System.IO.MemoryStream                          m_streamEntry = null;


public ZipOutputStream(System.IO.Stream archive)
{
    if (archive.CanWrite == false) {
        throw new System.InvalidOperationException();
    }
    archive.Position = 0;
    m_theArchive = archive;
}


public void PutNextEntry(ZipEntry entry)
{
    if (m_finished) {
        throw new System.InvalidOperationException();
    }
    if (m_streamEntry != null) {
        this.CloseEntry();
    }
    m_infoEntry = entry;
    m_streamEntry = new System.IO.MemoryStream();
}


public override void Write(byte[] buffer, int offset, int count)
{
    m_streamEntry.Write(buffer, offset, count);
}


public override void WriteByte(byte b)
{
    m_streamEntry.WriteByte(b);
}


public override bool CanWrite
{
    get {
        return true;
    }
}


public void Finish()
{
    if (m_finished) {
        return;
    }

    m_finished = true;
    if (m_streamEntry != null) {
        this.CloseEntry();
    }
    this.WriteCentralDirectoryStructure();
}


public void CloseEntry()
{
    if (m_streamEntry == null) {
        throw new System.InvalidOperationException();
    }

    ZipEntryWriter entry = new ZipEntryWriter(m_theArchive, m_infoEntry, m_streamEntry);
    m_compressedEntries.Add(entry);

    m_streamEntry.Close();
    m_streamEntry = null;
    m_infoEntry = null;
}


public bool IsFinished
{
    get {
        return m_finished;
    }
}


public override void Close()
{
    this.Finish();
    m_theArchive.Close();
}


private void WriteCentralDirectoryStructure()
{
    // the central directory structure
    long start = m_theArchive.Length;
    foreach (ZipEntryWriter entry in m_compressedEntries) {
        entry.WriteCentralDirectoryStructureFileHeader(m_theArchive);
    }
    long end = m_theArchive.Length;
    byte[] record = BuildEndOfCentralDirectoryRecord(m_compressedEntries.Count, start, end);
    m_theArchive.Write(record, 0, record.Length);
}


// I.  End of central directory record:
//
//       end of central dir signature    4 bytes  (0x06054b50)
//       number of this disk             2 bytes
//       number of the disk with the
//       start of the central directory  2 bytes
//       total number of entries in the
//       central directory on this disk  2 bytes
//       total number of entries in
//       the central directory           2 bytes
//       size of the central directory   4 bytes
//       offset of start of central
//       directory with respect to
//       the starting disk number        4 bytes
//       .ZIP file comment length        2 bytes
//       .ZIP file comment       (variable size)
//
private static byte[] BuildEndOfCentralDirectoryRecord(int qtyEntries, long offsetStart, long offsetEnd)
{
    byte[] record = new byte[22];

    // end of central dir signature    4 bytes  (0x06054b50)
    record[0] = 0x50;
    record[1] = 0x4b;
    record[2] = 0x05;
    record[3] = 0x06;

    // number of this disk             2 bytes
    record[4] = 0;
    record[5] = 0;

    // number of the disk with the start of the central directory  2 bytes
    record[6] = 0;
    record[7] = 0;

    // total number of entries in the central directory on this disk  2 bytes
    record[8] = (byte)  (qtyEntries & 0x00ff);
    record[9] = (byte) ((qtyEntries & 0xff00) >> 8);

    // total number of entries in the central directory           2 bytes
    record[10]= record[8];
    record[11]= record[9];

    // size of the central directory   4 bytes
    byte[] cc = System.BitConverter.GetBytes((int) (offsetEnd - offsetStart));
    record[12] = cc[0];
    record[13] = cc[1];
    record[14] = cc[2];
    record[15] = cc[3];

    // offset of start of central directory with respect to the starting disk number 4 bytes
    byte[] c = System.BitConverter.GetBytes((int) offsetStart);
    record[16] = c[0];
    record[17] = c[1];
    record[18] = c[2];
    record[19] = c[3];

    // .ZIP file comment length        2 bytes
    record[20] = 0x00;
    record[21] = 0x00;

    return record;
}


public override void Flush()
{
    m_theArchive.Flush();
}


public override bool CanRead
{
    get {
        return false;
    }
}


public override int Read(byte[] b, int offset, int count)
{
    throw new System.NotSupportedException();
}


public override bool CanSeek
{
    get {
        return false;
    }
}


public override long Seek(long offset, System.IO.SeekOrigin origin)
{
    throw new System.NotSupportedException();
}


public override long Length
{
    get {
        return m_theArchive.Length;
    }
}


public override long Position
{
    get {
        return m_theArchive.Position;
    }
    set {
        throw new System.NotSupportedException();
    }
}

public override void SetLength(long val)
{
    throw new System.NotSupportedException();
}


public void SetLevel(int level)
{
    throw new System.NotSupportedException();
}


public int GetLevel()
{
    throw new System.NotSupportedException();
}


public void SetComment(string comment)
{
    throw new System.NotSupportedException();
}

// ==================================================================================================================================
}//class
}//ns

