/* ***** BEGIN LICENSE BLOCK *****
 * Version: MPL 1.1/GPL 2.0/LGPL 2.1
 *
 * The contents of this file are subject to the Mozilla Public License Version
 * 1.1 (the "License"); you may not use this file except in compliance with
 * the License. You may obtain a copy of the License at
 * http://www.mozilla.org/MPL/
 *
 * Software distributed under the License is distributed on an "AS IS" basis,
 * WITHOUT WARRANTY OF ANY KIND, either express or implied. See the License
 * for the specific language governing rights and limitations under the
 * License.
 *
 * The Original Code is hultmann localization tools.
 *
 * The Initial Developer of the Original Code is
 * Jeferson Hultmann <hultmann@gmail.com>
 * Portions created by the Initial Developer are Copyright (C) 2005
 * the Initial Developer. All Rights Reserved.
 *
 * Contributor(s):
 *
 * Alternatively, the contents of this file may be used under the terms of
 * either of the GNU General Public License Version 2 or later (the "GPL"),
 * or the GNU Lesser General Public License Version 2.1 or later (the "LGPL"),
 * in which case the provisions of the GPL or the LGPL are applicable instead
 * of those above. If you wish to allow use of your version of this file only
 * under the terms of either the GPL or the LGPL, and not to allow others to
 * use your version of this file under the terms of the MPL, indicate your
 * decision by deleting the provisions above and replace them with the notice
 * and other provisions required by the GPL or the LGPL. If you do not delete
 * the provisions above, a recipient may use your version of this file under
 * the terms of any one of the MPL, the GPL or the LGPL.
 *
 * ***** END LICENSE BLOCK ***** */

using System.Collections.Generic;
using System.Text.RegularExpressions;

namespace LocaleInpector
{

internal sealed class MergeProperties : Merge
{

private string m_value;

public MergeProperties(PropertiesFile dic) : base(dic)
{
}


protected override string MergeFile(string translatedModule, string path)
{
    PropertiesFile fileTranslated = new PropertiesFile(translatedModule + path);
    Dictionary<string, string> translation = fileTranslated.LoadHashtable();
    return this.MergeFileEx(translation, path);
}


protected override string MergeFileEx(Dictionary<string, string> translation, string path)
{
    List<string> linesOriginal = Strings.SplitLines(m_dic.Content);
    string[] linesTranslated = new string[linesOriginal.Count];
    Dictionary<string, string> keysSource = m_dic.LoadHashtable(); // h:wtf
    MatchEvaluator myDelegate = new MatchEvaluator(this.MergeItemCallback);

    for (int idx = 0; idx < linesOriginal.Count; idx++) {
        string myLineOriginal = linesOriginal[idx];
        string[] keyvalueOriginal = PropertiesFile.ParsePropertyLine(myLineOriginal);
        if (keyvalueOriginal.Length == 1) {
            linesTranslated[idx] = string.Empty; // wrapped value / comment
            continue;
        }

        string myKey = keyvalueOriginal[0];
        if (myKey == "#") {
            linesTranslated[idx] = myLineOriginal; // comment
            continue;
        }

        if (translation.ContainsKey(myKey) == false) {
            // myKey no existe na localization
            this.WriteWarning("missing " + path + " => " + myKey + ". Using the original value.");
            linesTranslated[idx] = myLineOriginal;
        } else {
            // replace
            m_value = translation[myKey];
            linesTranslated[idx] = Regex.Replace(myLineOriginal, @"=(\s*).*", myDelegate);

            string err = TestPropertiesString(myLineOriginal, linesTranslated[idx]); // bug: test includes full line
            if (err != null) {
                linesTranslated[idx] = myLineOriginal;
                this.WriteWarning(err + ": " + path + " => " + myKey + ". Using the original value.");
            }
        }
    }

    return string.Join(m_dic.Eol, linesTranslated);
}


private string MergeItemCallback(Match match)
{
    GroupCollection groups = match.Groups;
    return "=" + groups[1].Value + this.Escape(m_value);
}


private static string TestPropertiesString(string o, string t)
{
    string err;
    err = CStyleFormatTest.CompareCore(o, t);
    if (err != null) {
        return err;
    }
    err = PlaceholderExpressionsTest.CompareCore(o, t);
    if (err != null) {
        return err;
    }
    return err;
}


// http://www.mozilla.org/projects/l10n/mlp_chrome.html#text
//InvalidImage=The image \u201c%S\u201d cannot be displayed, because it contains errors.
private string Escape(string txt)
{
    bool escaped = Config2.Current.GetPref("escaped") == "true";


    System.Text.StringBuilder buffer = new System.Text.StringBuilder(txt);
    if (m_dic.Path.EndsWith(".properties")) {

        for (int idx = buffer.Length - 1; idx > -1; idx--) {

            char ch = buffer[idx];
            switch (ch) {
                case '\r':
                case '\n':
                    break;
                case '"':
                case '\'':
                    buffer.Insert(idx, '\\');
                    break;
                //   color=\042#808080\042
                //   color=\\042#808080\\042
                case '\\':
                    buffer.Insert(idx, '\\');
                    break;
                default:
                    if ((ch < 0x20) || (escaped && (ch > 0x7f))) {
                        buffer[idx] = '\\';
                        buffer.Insert(idx + 1, "u" + ((int) ch).ToString("x4"));
                    }
                    break;
             }
        }
    } else {
        // mail/installer/installer.inc
    }

    buffer = buffer.Replace("\r\n", @"\n");
    buffer = buffer.Replace("\r",   @"\n");
    buffer = buffer.Replace("\n",   @"\n");

    return buffer.ToString();
}

}//class
}//ns
