/**
 * <copyright> 
 * 
 * Copyright (c) 2004-2005 IBM Corporation and others. 
 * All rights reserved. This program and the accompanying materials 
 * are made available under the terms of the Eclipse Public License - v 1.0 
 * which accompanies this distribution, and is available at 
 * http://opensource.org/licenses/eclipse-1.0.txt 
 * 
 * Contributors: 
 *   IBM - Initial API and implementation 
 * 
 * </copyright> 
 * 
 * $Id: TBox.java,v 1.1 2007/03/18 09:06:59 lzhang Exp $
 */

package org.eclipse.eodm.owl.reasoner.structural;

import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Map;

/**
 * 
 * TBox
 *   Structural Reasoner 
 */

public class TBox {
    
	public TBox(RBox rbox) {
		this.rbox = rbox;
		
		int thingID = getConceptID(ReasonerUtil.TOP, false, true);
		int nothingID = getConceptID(ReasonerUtil.BOTTOM, false, true);
		
		thingNode = new CTNode(this, thingID);
		nothingNode = new CTNode(this, nothingID);
		
		addNodeContainment(thingNode, nothingNode);
		
		new AllValueFromLink(this, thingNode, thingNode, rbox.getTopPropertyID());
		new SomeValueFromLink(this, nothingNode, nothingNode, rbox.getTopPropertyID());
		
	}
	
	/**
	 *  add concept into tbox
	 */
    public void addConcept(Concept c) {
        classify(c);
    }
       
    
    /**
     *  
     */
    public void addSubOfAxiom(Concept child, Concept father) {
        CTNode childNode = classify(child);
        CTNode fatherNode = classify(father);
        
        addNodeContainment(fatherNode,childNode);
        
    }
    
    public void addEquivalentAxiom(Concept c1, Concept c2) {
    	CTNode n1 = classify(c1);
    	CTNode n2 = classify(c2);
    	
    	addNodeContainment(n1,n2);
    	addNodeContainment(n2,n1);
    }
    
    public void addDisjoint(Concept c1, Concept c2) {
    	Intersection in = TermFactory.instance().createIntersect(c1,c2);
    	addSubOfAxiom(in, ReasonerUtil.BOTTOM);
    }
    
    public RBox getRBox() {
    	return rbox;
    }
    
    /**
     * Classify a concept C into the classification tree
     * @param c
     */
    CTNode classify(Concept c) {
   	
        // normalize
    	Concept[] forms = ReasonerUtil.getAllForms(c,rbox);
    	
    	CTNode n = getNodeByConcept(forms[0],false);
    	
    	if(n!=null){
    		return n;
    	}
    	
    	// create a concept ID for concept 
    	int cid = getConceptID(forms[0],false, true);
    	
    	for(int i=0; i<forms.length;i++){
    		if(!exprSet.contains(forms[i])){
    			exprSet.add(forms[i]);
        		n = classify(forms[i], cid, n);
    		}
    	}
    	
    	return n;
    }
    
    /**
     * Add c whose id is cID into given node, and reclassify the node  
     * If given node is null, then create a new node 
     * 
     * @param c  concept to be classified 
     * @param cID    
     * @param node
     * @return
     */
    
    CTNode classify(Concept c, int cID, CTNode node) {   	
    	List superClassList = new ArrayList();
		superClassList.add(thingNode);
		List subClassList = new ArrayList();
		subClassList.add(nothingNode);
		
		if(node == null){
			node = new CTNode(this,cID);
		} else {
			node.addConceptID(cID);
		}
		
		switch( c.getCType() ){
		case TermUtil.C_INTERSECT:
			List inNodeList = new ArrayList();
			for(Iterator it=((Intersection) c).getIntersectOf().iterator();it.hasNext();) {
				CTNode v = classify((Concept) it.next());
				inNodeList.add(v);
			}
			
			for (int i = inNodeList.size() - 1; i >= 0; i--) {
				CTNode v1 = (CTNode) inNodeList.get(i);
				for (int j = i - 1; j >= 0; j--) {
					CTNode v2 = (CTNode) inNodeList.get(j);
					if (isSubNodeOf(v1, v2)) {
						inNodeList.remove(j);
						i--;
					} else if (isSubNodeOf(v2, v1)) {
						inNodeList.remove(i);
						break;
					}
				}
			}
			superClassList.addAll(inNodeList);
			subClassList.addAll(getSubNodes(inNodeList));
			
			int targetID = getNextTargetID();
			for (int i = 0; i < inNodeList.size(); i++) {
				new ConjunctionLink(this, ((CTNode) inNodeList.get(i))
						.getAlternativeNode(), node.getAlternativeNode(),
						targetID);
			}
			targetValue[targetID] = inNodeList.size();
			break;
			
		case TermUtil.C_UNION:
			List unNodeList = new ArrayList();
			for(Iterator it= ((Union)c).getUnionOf().iterator();it.hasNext();){
				CTNode v = classify((Concept) it.next());
				unNodeList.add(v);
			}
			
			for (int i = unNodeList.size() - 1; i >= 0; i--) {
				CTNode v1 = (CTNode) unNodeList.get(i);
				for (int j = i - 1; j >= 0; j--) {
					CTNode v2 = (CTNode) unNodeList.get(j);
					if (isSubNodeOf(v2, v1)) {
						unNodeList.remove(j);
						i--;
					} else if (isSubNodeOf(v1, v2)) {
						unNodeList.remove(i);
						break;
					}
				}
			}
			
			subClassList.addAll(unNodeList);
			superClassList.addAll(getSuperNodes(unNodeList));
			targetID = getNextTargetID();
			for (int i = 0; i < unNodeList.size(); i++) {
				new DisjunctionLink(this, ((CTNode) unNodeList.get(i))
						.getAlternativeNode(), node.getAlternativeNode(),
						targetID);
			}
			targetValue[targetID] = unNodeList.size();
			break;
			
		case TermUtil.C_SOME:
			int pID = rbox.getCreatePropertyID(((SomeValueFrom)c).getOnProperty());

			CTNode v = classify( ((SomeValueFrom) c).getRange() );

			new SomeValueFromLink(this, node.getAlternativeNode(), v.getAlternativeNode(), pID);
			for (Iterator iter = getSuperNodes(v).iterator(); iter.hasNext();) {
				Iterator iter1 = ((CTNode) iter.next()).someValuesFromByLinks.iterator();
				while (iter1.hasNext()) {
					SomeValueFromLink link = (SomeValueFromLink) iter1.next();
					if (rbox.isSubPropertyOf(pID, link.propertyID)) {
						superClassList.add(link.getSource());
					}
				}
			}
			
			for (Iterator iter = getSubNodes(v).iterator(); iter.hasNext();) {
				Iterator iter1 = ((CTNode) iter.next()).someValuesFromByLinks.iterator();
				while (iter1.hasNext()) {
					SomeValueFromLink link = (SomeValueFromLink) iter1.next();
					if (rbox.isSubPropertyOf(link.propertyID, pID)) {
						subClassList.add(link.getSource());
					}
				}
			}
			break;
			
		case TermUtil.C_ALL:
			pID = rbox.getCreatePropertyID(((AllValueFrom) c).getOnProperty());
			v = classify( ((AllValueFrom) c).getRange());
			new AllValueFromLink(this, node.getAlternativeNode(), v.getAlternativeNode(), pID);
			for (Iterator iter = getSuperNodes(v).iterator(); iter.hasNext();) {
				Iterator iter1 = ((CTNode) iter.next()).allValuesFromByLinks.iterator();
				while (iter1.hasNext()) {
					AllValueFromLink link = (AllValueFromLink) iter1.next();
					if (rbox.isSubPropertyOf(link.propertyID, pID)) {
						superClassList.add(link.getSource());
					}
				}
			}
			
			for (Iterator iter = getSubNodes(v).iterator(); iter.hasNext();) {
				Iterator iter1 = ((CTNode) iter.next()).allValuesFromByLinks.iterator();
				while (iter1.hasNext()) {
					AllValueFromLink link = (AllValueFromLink) iter1.next();
					if (rbox.isSubPropertyOf(pID, link.propertyID)) {
						subClassList.add(link.getSource());
					}
				}
			}
			break;
			
		case TermUtil.C_MIN:
		case TermUtil.C_CARD:
		case TermUtil.C_MAX:
        case TermUtil.C_NEGATION:
        default:
			
		}
		
		for (Iterator iter = superClassList.iterator(); iter.hasNext();) {
			addNodeContainment((CTNode) iter.next(), node);
		}
		for (Iterator iter = subClassList.iterator(); iter.hasNext();) {
			addNodeContainment(node, (CTNode) iter.next());
		}
		
    	return node.getAlternativeNode();
    }
    
	public List getSubNodes(CTNode v) {
		List result = new ArrayList();
		int p = v.getAlternativeNode().topologicalPosition;
		for (int i = nodeList.size() - 1; i >= p; i--) {
			flags[i] = 0;
		}
		flags[p] = 1;
		for (int i = p; i < nodeList.size(); i++) {
			if (flags[i] == 1) {
				CTNode v1 = (CTNode) nodeList.get(i);
				result.add(v1);
				Iterator iter = v1.containmentLinks.iterator();
				while (iter.hasNext()) {
					flags[((ContainmentLink) iter.next()).getTarget().topologicalPosition] = 1;
				}
			}
		}
		return result;
	}
    
	public List getSuperNodes(CTNode v) {
		List result = new ArrayList();
		int p = v.getAlternativeNode().topologicalPosition;
		for (int i = 0; i <= p; i++) {
			flags[i] = 0;
		}
		flags[p] = 1;
		for (int i = p; i >= 0; i--) {
			if (flags[i] == 1) {
				CTNode v1 = (CTNode) nodeList.get(i);
				result.add(v1);
				Iterator iter = v1.containmentByLinks.iterator();
				while (iter.hasNext()) {
					flags[((ContainmentLink) iter.next()).getSource().topologicalPosition] = 1;
				}
			}
		}
		return result;
	}
    
	public List getSubNodes(Collection c) {
		List result = new ArrayList();
		if (c.size() <= 31) {
			for (int i = nodeList.size() - 1; i >= 0; i--) {
				flags[i] = 0;
			}
			Iterator iter = c.iterator();
			int index = 0;
			int min = nodeList.size();
			while (iter.hasNext()) {
				int p = ((CTNode) iter.next()).getAlternativeNode().topologicalPosition;
				flags[p < min ? min = p : p] |= 1 << (index++);
			}
			int target = (1 << index) - 1;
			for (int i = min; i < nodeList.size(); i++) {
				if (flags[i] > 0) {
					CTNode v1 = (CTNode) nodeList.get(i);
					if (flags[i] == target) {
						result.add(v1);
					}
					iter = v1.containmentLinks.iterator();
					while (iter.hasNext()) {
						flags[((ContainmentLink) iter.next()).getTarget().topologicalPosition] |= flags[i];
					}
				}
			}
		} else {
			for (int i = nodeList.size() - 1; i >= 0; i--) {
				flags[i] = 1;
			}
			Iterator iter = c.iterator();
			int flags1[] = new int[nodeList.size()];
			while (iter.hasNext()) {
				for (int i = 0; i < flags1.length; i++) {
					flags1[i] = 0;
				}
				int p = ((CTNode) iter.next()).getAlternativeNode().topologicalPosition;
				flags1[p] = 1;
				for (int i = 0; i < flags1.length; i++) {
					if (flags1[i] == 1) {
						Iterator iter1 = ((CTNode) nodeList.get(i)).containmentLinks
								.iterator();
						while (iter1.hasNext()) {
							flags1[((ContainmentLink) iter1.next()).getTarget().topologicalPosition] = 1;
						}
					} else {
						flags[i] = 0;
					}
				}
			}
			for (int i = 0; i < nodeList.size(); i++) {
				if (flags[i] == 1) {
					result.add(nodeList.get(i));
				}
			}
		}
		return result;
	}
    
	public List getSuperNodes(Collection c) {
		List result = new ArrayList();
		if (c.size() <= 31) {
			for (int i = nodeList.size() - 1; i >= 0; i--) {
				flags[i] = 0;
			}
			Iterator iter = c.iterator();
			int index = 0;
			int max = -1;
			while (iter.hasNext()) {
				int p = ((CTNode) iter.next()).getAlternativeNode().topologicalPosition;
				flags[p > max ? max = p : p] |= 1 << (index++);
			}
			int target = (1 << index) - 1;
			for (int i = max; i >= 0; i--) {
				if (flags[i] > 0) {
					CTNode v1 = (CTNode) nodeList.get(i);
					if (flags[i] == target) {
						result.add(v1);
					}
					iter = v1.containmentByLinks.iterator();
					while (iter.hasNext()) {
						flags[((ContainmentLink) iter.next()).getSource().topologicalPosition] |= flags[i];
					}
				}
			}
		} else {
			for (int i = nodeList.size() - 1; i >= 0; i--) {
				flags[i] = 1;
			}
			Iterator iter = c.iterator();
			int flags1[] = new int[nodeList.size()];
			while (iter.hasNext()) {
				for (int i = 0; i < flags1.length; i++) {
					flags1[i] = 0;
				}
				int p = ((CTNode) iter.next()).getAlternativeNode().topologicalPosition;
				flags1[p] = 1;
				for (int i = flags1.length - 1; i >= 0; i--) {
					if (flags1[i] == 1) {
						Iterator iter1 = ((CTNode) nodeList.get(i)).containmentByLinks
								.iterator();
						while (iter1.hasNext()) {
							flags1[((ContainmentLink) iter1.next()).getTarget().topologicalPosition] = 1;
						}
					} else {
						flags[i] = 0;
					}
				}
			}
			for (int i = 0; i < nodeList.size(); i++) {
				if (flags[i] == 1) {
					result.add(nodeList.get(i));
				}
			}
		}
		return result;
	}
	
	private List getSubClassNodesWithRemoveRedundancy(CTNode v) {
		List result = new ArrayList();
		int p = v.getAlternativeNode().topologicalPosition;
		for (int i = nodeList.size() - 1; i >= p; i--) {
			flags[i] = 0;
		}
		flags[p] = 1;
		ContainmentLink[] tempArray = new ContainmentLink[nodeList.size()];
		for (int i = p; i < nodeList.size(); i++) {
			if (flags[i] == 1) {
				CTNode v1 = (CTNode) nodeList.get(i);
				result.add(v1);
				Iterator iter = v1.containmentLinks.iterator();
				if (i == p) {
					List removeList = new ArrayList();
					while (iter.hasNext()) {
						ContainmentLink link = (ContainmentLink) iter.next();
						int target = link.getTarget().topologicalPosition;
						if (flags[target] == 1) {
							removeList.add(link);
						} else {
							flags[target] = 1;
							tempArray[target] = link;
						}
					}
					for (int j = 0; j < removeList.size(); j++) {
						ContainmentLink link = ((ContainmentLink) removeList
								.get(j));
						link.setSource(null);
						link.setTarget(null);
					}
				} else {
					while (iter.hasNext()) {
						int target = ((ContainmentLink) iter.next())
								.getTarget().topologicalPosition;
						flags[target] = 1;
						if (tempArray[target] != null) {
							tempArray[target].setSource(null);
							tempArray[target].setTarget(null);
							tempArray[target] = null;
						}
					}
				}
			}
		}
		return result;
	}

	private List getSuperClassNodesWithRemoveRedundancy(CTNode v) {
		List result = new ArrayList();
		int p = v.getAlternativeNode().topologicalPosition;
		for (int i = 0; i <= p; i++) {
			flags[i] = 0;
		}
		flags[p] = 1;
		ContainmentLink[] tempArray = new ContainmentLink[p + 1];
		for (int i = p; i >= 0; i--) {
			if (flags[i] == 1) {
				CTNode v1 = (CTNode) nodeList.get(i);
				result.add(v1);
				Iterator iter = v1.containmentByLinks.iterator();
				if (i == p) {
					List removeList = new ArrayList();
					while (iter.hasNext()) {
						ContainmentLink link = (ContainmentLink) iter.next();
						int source = link.getSource().topologicalPosition;
						if (flags[source] == 1) {
							removeList.add(link);
						} else {
							flags[source] = 1;
							tempArray[source] = link;
						}
					}
					for (int j = 0; j < removeList.size(); j++) {
						ContainmentLink link = ((ContainmentLink) removeList
								.get(j));
						link.setSource(null);
						link.setTarget(null);
					}
				} else {
					while (iter.hasNext()) {
						int source = ((ContainmentLink) iter.next())
								.getSource().topologicalPosition;
						flags[source] = 1;
						if (tempArray[source] != null) {
							tempArray[source].setSource(null);
							tempArray[source].setTarget(null);
							tempArray[source] = null;
						}
					}
				}
			}
		}
		return result;
	}
	
	public CTNode findNextSubClassDisjunction(CTNode v) {
		int p = v.getAlternativeNode().topologicalPosition;
		for (int i = nodeList.size() - 1; i >= p; i--) {
			flags[i] = 0;
		}
		for (int i = 0; i < targetCounter.length; i++) {
			targetCounter[i] = 0;
		}
		List tempList = new ArrayList();
		flags[p] = 1;
		for (int i = p; i < nodeList.size(); i++) {
			if (flags[i] == 1) {
				CTNode v1 = (CTNode) nodeList.get(i);
				Iterator iter = v1.containmentLinks.iterator();
				while (iter.hasNext()) {
					flags[((ContainmentLink) iter.next()).getTarget().topologicalPosition] = 1;
				}
				iter = v1.disjunctionLinks.iterator();
				while (iter.hasNext()) {
					DisjunctionLink link = (DisjunctionLink) iter.next();
					if (targetValue[link.getTargetID()] == (++targetCounter[link
							.getTargetID()])) {
						if (flags[link.getTarget().topologicalPosition] == 0) {
							tempList.add(link.getTarget());
						}
					}
				}
			}
		}
		for (Iterator iter = tempList.iterator(); iter.hasNext();) {
			CTNode v1 = (CTNode) iter.next();
			if (flags[v1.topologicalPosition] == 0) {
				return v1;
			}
		}
		return null;
	}

	public CTNode findNextSuperClassConjunction(CTNode v) {
		int p = v.getAlternativeNode().topologicalPosition;
		for (int i = 0; i <= p; i++) {
			flags[i] = 0;
		}
		for (int i = 0; i < targetCounter.length; i++) {
			targetCounter[i] = 0;
		}
		List tempList = new ArrayList();
		flags[p] = 1;
		for (int i = p; i >= 0; i--) {
			if (flags[i] == 1) {
				CTNode v1 = (CTNode) nodeList.get(i);
				Iterator iter = v1.containmentByLinks.iterator();
				while (iter.hasNext()) {
					flags[((ContainmentLink) iter.next()).getSource().topologicalPosition] = 1;
				}
				iter = v1.conjunctionLinks.iterator();
				while (iter.hasNext()) {
					ConjunctionLink link = (ConjunctionLink) iter.next();
					if (targetValue[link.getTargetID()] == (++targetCounter[link
							.getTargetID()])) {
						if (flags[link.getTarget().topologicalPosition] == 0) {
							tempList.add(link.getTarget());
						}
					}
				}
			}
		}
		for (Iterator iter = tempList.iterator(); iter.hasNext();) {
			CTNode v1 = (CTNode) iter.next();
			if (flags[v1.topologicalPosition] == 0) {
				return v1;
			}
		}
		return null;
	}
	
    void addNodeContainment(CTNode father, CTNode child) {
		if (isSubNodeOf(child, father)) {
			return;
		}

		adjustNodeList(father, child);

		father = father.getAlternativeNode();
		child = child.getAlternativeNode();
		ArrayList tempList = new ArrayList();
		List superClassList;
		List subClassList;
		if (father != child) {
			superClassList = getSuperNodes(father);
			subClassList = getSubNodes(child);
			for (int i = nodeList.size() - 1; i >= 0; i--) {
				flags[i] = 0;
			}
			for (Iterator iter = subClassList.iterator(); iter.hasNext();) {
				flags[((CTNode) iter.next()).topologicalPosition] = 1;
			}
			for (Iterator iter = superClassList.iterator(); iter.hasNext();) {
				for (Iterator iter1 = ((CTNode) iter.next()).containmentLinks
						.iterator(); iter1.hasNext();) {
					ContainmentLink link = (ContainmentLink) iter1.next();
					if (flags[link.getTarget().topologicalPosition] == 1) {
						tempList.add(link);
					}
				}
			}
			new ContainmentLink(this, father, child);

		} else {
			superClassList = getSuperClassNodesWithRemoveRedundancy(father);
			subClassList = getSubClassNodesWithRemoveRedundancy(child);
			for (int i = nodeList.size() - 1; i >= 0; i--) {
				flags[i] = 0;
			}
			for (Iterator iter = subClassList.iterator(); iter.hasNext();) {
				flags[((CTNode) iter.next()).topologicalPosition] = 1;
			}
			for (Iterator iter = superClassList.iterator(); iter.hasNext();) {
				for (Iterator iter1 = ((CTNode) iter.next()).containmentLinks
						.iterator(); iter1.hasNext();) {
					ContainmentLink link = (ContainmentLink) iter1.next();
					if (flags[link.getTarget().topologicalPosition] == 1
							&& link.getSource() != father && link.getTarget() != child) {
						tempList.add(link);
					}
				}
			}
		}

		for (Iterator iter = tempList.iterator(); iter.hasNext();) {
			ContainmentLink link = (ContainmentLink) iter.next();
			link.setSource(null);
			link.setTarget(null);
		}
		HashMap superMap = new HashMap();
		HashMap subMap = new HashMap();
		for (Iterator iter = superClassList.iterator(); iter.hasNext();) {
			for (Iterator iter1 = ((CTNode) iter.next()).allValuesFromByLinks
					.iterator(); iter1.hasNext();) {
				AllValueFromLink link = (AllValueFromLink) iter1.next();
				Integer key = new Integer(link.getPropertyID());
				if (!superMap.containsKey(key)) {
					superMap.put(key, new ArrayList());
				}
				((ArrayList) superMap.get(key)).add(link.getSource());
			}
		}
		for (Iterator iter = subClassList.iterator(); iter.hasNext();) {
			for (Iterator iter1 = ((CTNode) iter.next()).allValuesFromByLinks
					.iterator(); iter1.hasNext();) {
				AllValueFromLink link = (AllValueFromLink) iter1.next();
				Integer key = new Integer(link.getPropertyID());
				if (!subMap.containsKey(key)) {
					subMap.put(key, new ArrayList());
				}
				((ArrayList) subMap.get(key)).add(link.getSource());
			}
		}
		for (Iterator superIter = superMap.entrySet().iterator(); superIter
				.hasNext();) {
			Map.Entry superEntry = (Map.Entry) superIter.next();
			for (Iterator subIter = subMap.entrySet().iterator(); subIter
					.hasNext();) {
				Map.Entry subEntry = (Map.Entry) subIter.next();
				if (rbox.isSubPropertyOf(((Integer) superEntry.getKey())
						.intValue(), ((Integer) subEntry.getKey()).intValue())) {
					for (Iterator iter = ((ArrayList) superEntry.getValue())
							.iterator(); iter.hasNext();) {
						CTNode superV = (CTNode) iter.next();
						for (Iterator iter1 = ((ArrayList) subEntry.getValue())
								.iterator(); iter1.hasNext();) {
							CTNode subV = (CTNode) iter1.next();
							addNodeContainment(superV, subV);
						}
					}
				}
			}
		}
		superMap.clear();
		subMap.clear();

		for (Iterator iter = superClassList.iterator(); iter.hasNext();) {
			for (Iterator iter1 = ((CTNode) iter.next()).someValuesFromByLinks
					.iterator(); iter1.hasNext();) {
				SomeValueFromLink link = (SomeValueFromLink) iter1.next();
				Integer key = new Integer(link.getPropertyID());
				if (!superMap.containsKey(key)) {
					superMap.put(key, new ArrayList());
				}
				((ArrayList) superMap.get(key)).add(link.getSource());
			}
		}
		for (Iterator iter = subClassList.iterator(); iter.hasNext();) {
			for (Iterator iter1 = ((CTNode) iter.next()).someValuesFromByLinks
					.iterator(); iter1.hasNext();) {
				SomeValueFromLink link = (SomeValueFromLink) iter1.next();
				Integer key = new Integer(link.getPropertyID());
				if (!subMap.containsKey(key)) {
					subMap.put(key, new ArrayList());
				}
				((ArrayList) subMap.get(key)).add(link.getSource());
			}
		}
		for (Iterator superIter = superMap.entrySet().iterator(); superIter
				.hasNext();) {
			Map.Entry superEntry = (Map.Entry) superIter.next();
			for (Iterator subIter = subMap.entrySet().iterator(); subIter
					.hasNext();) {
				Map.Entry subEntry = (Map.Entry) subIter.next();
				if (rbox
						.isSubPropertyOf(
								((Integer) subEntry.getKey()).intValue(),
								((Integer) superEntry.getKey()).intValue())) {
					for (Iterator iter = ((ArrayList) superEntry.getValue())
							.iterator(); iter.hasNext();) {
						CTNode superV = (CTNode) iter.next();
						for (Iterator iter1 = ((ArrayList) subEntry.getValue())
								.iterator(); iter1.hasNext();) {
							CTNode subV = (CTNode) iter1.next();
							addNodeContainment(superV, subV);
						}
					}
				}
			}
		}

		for (Iterator iter = superClassList.iterator(); iter.hasNext();) {
			CTNode v = (CTNode) iter.next();
			CTNode vv = findNextSubClassDisjunction(v);
			if (vv != null) {
				addNodeContainment(v, vv);
			}
		}

		for (Iterator iter = subClassList.iterator(); iter.hasNext();) {
			CTNode v = (CTNode) iter.next();
			CTNode vv = findNextSuperClassConjunction(v);
			if (vv != null) {
				addNodeContainment(vv, v);
			}
		}

    	
    }
    
	private void combineNode(CTNode v1, CTNode v2) {
		if (v1.tbox != this || v2.tbox != this) {
			throw new IllegalArgumentException();
		}
		v1 = v1.getAlternativeNode();
		v2 = v2.getAlternativeNode();
		
		if (v1 != v2) {
			for(Iterator it = v2.cIDSet.iterator();it.hasNext();){
				Integer id = (Integer) it.next();
				v1.cIDSet.add(id);
				cid2NodeMap.put(id, v1);
			}
			v2.cIDSet.clear();

			for(Iterator it = v2.containmentLinks.iterator();it.hasNext();){
				ContainmentLink link = (ContainmentLink) it.next();
				v1.containmentLinks.add(link);
				link.source = v1;
			}

			for(Iterator it = v2.containmentByLinks.iterator();it.hasNext();){
				ContainmentLink link = (ContainmentLink) it.next();
				v1.containmentByLinks.add(link);
				link.target = v1;
			}

			for(Iterator it = v2.conjunctionLinks.iterator();it.hasNext();){
				ConjunctionLink link = (ConjunctionLink) it.next();
				v1.conjunctionLinks.add(link);
				link.source = v1;
			}

			for(Iterator it = v2.disjunctionLinks.iterator();it.hasNext();){
				DisjunctionLink link = (DisjunctionLink) it.next();
				v1.disjunctionLinks.add(link);
				link.source = v1;
			}

			for(Iterator it = v2.allValuesFromByLinks.iterator();it.hasNext();){
				AllValueFromLink link = (AllValueFromLink) it.next();
				v1.allValuesFromByLinks.add(link);
				link.target = v1;
			}

			for(Iterator it = v2.someValuesFromByLinks.iterator();it.hasNext();){
				SomeValueFromLink link = (SomeValueFromLink) it.next();
				v1.someValuesFromByLinks.add(link);
				link.target = v1;
			}

			v2.alternativeNode = v1;
			if (v2 == thingNode) {
				thingNode = v1;
			}
			if (v2 == nothingNode) {
				nothingNode = v1;
			}
		}
	}
    
	private void adjustNodeList(CTNode v1, CTNode v2) {
		int pa = v2.getAlternativeNode().topologicalPosition;
		int pb = v1.getAlternativeNode().topologicalPosition;
		if (pa >= pb) {
			return;
		}
		for (int i = pa; i <= pb; i++) {
			flags[i] = 0;
		}
		flags[pa] = 1;
		for (int i = pa; i <= pb; i++) {
			if (flags[i] == 1) {
				Iterator iter = ((CTNode) nodeList.get(i)).containmentLinks
						.iterator();
				while (iter.hasNext()) {
					int p = ((ContainmentLink) iter.next()).getTarget().topologicalPosition;
					if (p <= pb) {
						flags[p] = 1;
					}
				}
			}
		}
		flags[pb] |= 2;
		for (int i = pb; i >= pa; i--) {
			if ((flags[i] & 2) == 2) {
				Iterator iter = ((CTNode) nodeList.get(i)).containmentByLinks
						.iterator();
				while (iter.hasNext()) {
					int p = ((ContainmentLink) iter.next()).getSource().topologicalPosition;
					if (p >= pa) {
						flags[p] |= 2;
					}
				}
			}
		}
		CTNode tempArray[] = new CTNode[pb - pa + 1];
		int index = 0;
		for (int i = pa; i <= pb; i++) {
			if (flags[i] == 2) {
				tempArray[index++] = (CTNode) nodeList.get(i);
			}
		}
		if (flags[pa] == 3) {
			CTNode v = (CTNode) nodeList.get(pa);
			for (int i = pa + 1; i <= pb; i++) {
				if (flags[i] == 3) {
					combineNode(v, (CTNode) nodeList.get(i));
				}
			}
			tempArray[index++] = v;
		}
		for (int i = pa; i <= pb; i++) {
			if (flags[i] <= 1) {
				tempArray[index++] = (CTNode) nodeList.get(i);
			}
		}
		for (int i = 0; i < index; i++) {
			nodeList.set(i + pa, tempArray[i]);
			tempArray[i].topologicalPosition = i + pa;
		}
		if (index < pb - pa + 1) {
			nodeList.subList(pa + index, pb + 1).clear();
			for (int i = nodeList.size() - 1; i >= pa + index; i--) {
				((CTNode) nodeList.get(i)).topologicalPosition = i;
			}
		}
	}
    
    /**
     * return true if n1 is sub node of n2
     */
    boolean isSubNodeOf(CTNode n1, CTNode n2){
		if (n1 == nothingNode || n2 == thingNode) {
			return true;
		}

		int pa = n2.getAlternativeNode().topologicalPosition;
		int pb = n1.getAlternativeNode().topologicalPosition;
		if (pa > pb) {
			return false;
		}
		for (int i = pa; i <= pb; i++) {
			flags[i] = 0;
		}
		flags[pa] = 1;
		for (int i = pa; i < pb; i++) {
			if (flags[i] == 1) {
				Iterator iter = ((CTNode) nodeList.get(i)).containmentLinks
						.iterator();
				while (iter.hasNext()) {
					flags[((ContainmentLink) iter.next()).getTarget().topologicalPosition] = 1;
				}
			}
		}
		
		return (flags[pb] == 1);

    }
    
	CTNode getNodeByConcept(Concept c, boolean doNormalizationFirst) {
		int cid = getConceptID(c, doNormalizationFirst,false);
		return getNodeByID(cid);
	}
    
	CTNode getNodeByID(int cID) {
		return (CTNode) cid2NodeMap.get(new Integer(cID));
	}
	
    void addNode(CTNode n) {
		n.topologicalPosition = nodeList.size();
		nodeList.add(n);
		if (nodeList.size() > flags.length) {
			flags = new int[(int) (nodeList.size() * 3 / 2)];
		}
    }
    
    /**
     * Get an ID for a concept. The ID for a concept is decided by the expression string.
     *  
     * @param c
     * @param doNormalizationFirst
     * @param create
     * @return ID
     */
    synchronized int getConceptID(Concept c, boolean doNormalizationFirst, boolean create) {
    	Concept term = c;
    	if(doNormalizationFirst) {
    		term = ReasonerUtil.toMinimalDNF(c,rbox); 
    	}
    	
    	String idStr = ReasonerUtil.toString(term);
    	Integer index = (Integer) idString2IDMap.get(idStr);
    	
    	if(index != null){
    		return index.intValue();
    	} else if(create){
    		idString2IDMap.put(idStr, new Integer(conceptList.size()) );
    		conceptList.add(ReasonerUtil.cloneConcept(c));
    		return conceptList.size() - 1;
    	} else {
    		return -1;
    	}
    	
    }
    
    synchronized Concept getConceptByID(int id) {
        if (id < 0 || id >= conceptList.size()) {
            return null;
        } else {
            return (Concept) conceptList.get(id);
            //ReasonerUtil.cloneConcept((Concept) conceptList.get(id));
        }
    }
    
    public synchronized int getConceptNum() {
        return conceptList.size();
    }
    
	private synchronized int getNextTargetID() {
		if (nextTargetID >= targetValue.length) {
			int newLength = nextTargetID * 3 / 2;
			int newArray[] = new int[newLength];
			System.arraycopy(targetValue, 0, newArray, 0, targetValue.length);
			targetValue = newArray;

			targetCounter = new int[newLength];
		}
		return nextTargetID++;
	}
    
    RBox rbox = null;
    
	private int nextTargetID = 0;
	
    private int targetValue[] = new int[10];
    private int targetCounter[] = new int[10];
    
    ArrayList conceptList = new ArrayList();
    
    ArrayList nodeList = new ArrayList();
    
    CTNode thingNode, nothingNode;
    
    int [] flags = new int[10];
    
    // get a node by a concept id
    HashMap cid2NodeMap = new HashMap(); 
    
    // 
    HashMap idString2IDMap = new HashMap();
    
    
    HashSet exprSet = new HashSet();
}
