/*******************************************************************************
 * Copyright (c) 2006 - 2016 CEA LIST.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License 2.0
 * which accompanies this distribution, and is available at
 * https://www.eclipse.org/legal/epl-2.0/
 * 
 * SPDX-License-Identifier: EPL-2.0
 * 
 * Contributors:
 *     Shuai Li (CEA LIST) <shuai.li@cea.fr> - initial API and implementation
 *     Ansgar Radermacher (CEA LIST) - bug 575604
 *******************************************************************************/

package org.eclipse.papyrus.designer.languages.java.codegen.xtend

import java.util.ArrayList
import java.util.List
import org.eclipse.emf.common.util.UniqueEList
import org.eclipse.papyrus.designer.languages.common.base.GenUtils
import org.eclipse.papyrus.designer.languages.java.codegen.utils.JavaGenUtils
import org.eclipse.papyrus.designer.languages.java.profile.PapyrusJava.External
import org.eclipse.uml2.uml.Classifier
import org.eclipse.uml2.uml.Element
import org.eclipse.uml2.uml.Enumeration
import org.eclipse.uml2.uml.Interface
import org.eclipse.uml2.uml.NamedElement
import org.eclipse.uml2.uml.Namespace
import org.eclipse.uml2.uml.Package
import org.eclipse.uml2.uml.TemplateParameterSubstitution
import org.eclipse.uml2.uml.UMLFactory
import org.eclipse.uml2.uml.util.UMLUtil

class JavaClassImportDeclaration {

	static def javaClassImports(NamedElement ne, Namespace ns) {
		var List<String> result = new ArrayList<String>()
		val import = JavaClassImportDeclaration.importName(ne, ns)
		if (import.length() > 0) {
			// bug 575604: don't add non-required imports to list (as prefix is added later)
			result.add(import)
		}
		return result
	}

	/**
	 * @param ne the element to import
	 * @param ns the element in which we import ne
	 * @returns the import or an empty string, if import is not required
	 */
	static def importName(NamedElement ne, Namespace ns) {
		// ne is owned by a template parameter substitution and it is not stereotyped external. This is an invalid model.
		if (ne.owner instanceof TemplateParameterSubstitution) {
			return ""
		}

		// Never import ne if its short name is the same as the name of an inner class directly owned by the current ns
		// We will use the full name of ne later on
		for (element : ns.ownedElements) {
			if (element instanceof Enumeration || element instanceof Interface ||
				element.eClass.equals(UMLFactory.eINSTANCE.getUMLPackage().getClass_())) {
				if ((element as Classifier).name.equals(ne.name)) {
					return ""
				}
			}
		}

		// Find if there is an import of ns, with the same class name and only import it if not so
		var classifier = ns
		if (!(ns.owner instanceof Package)) {
			classifier = getPackagedClassifier(ns)
			if (classifier === null) {
				return ""
			}
		}

		val qName = JavaGenUtils.javaQualifiedName(ne)
		// no import is required, if the referenced element is in the same namespace
		// or the qualified name is in java.lang
		if (ne.owner == ns.owner || JavaGenUtils.isJavaLang(qName)) {
			return ""
		}

		var importsOfClassifier = JavaGenUtils.imports.get(classifier)

		if (importsOfClassifier === null) {
			importsOfClassifier = new UniqueEList<String>()
			JavaGenUtils.imports.put(classifier, importsOfClassifier)
		} else {
			for (import : importsOfClassifier) {
				val String[] parts = import.split("\\.")
				if (parts.length > 0) {
					val String lastPart = parts.get(parts.length - 1)
					if (ne.name.equals(lastPart)) {
						return "" // Empty imports are not generated by JavaImportUtil.importDirective
					}
				}
			}
		}

		importsOfClassifier.add(qName)
		return qName
	}

	/**
	 * Get the nearest classifier following the owning relationship) or null, if there is none
	 */
	private static def Classifier getPackagedClassifier(Element ns) {
		if (ns instanceof Classifier) {
			return (ns as Classifier)
		} else {
			if (ns.owner === null) {
				return null
			}
			return getPackagedClassifier(ns.owner)
		}
	}
}
