/**
 * Copyright (c) 2018, 2020 Kiel University and others.
 * 
 * This program and the accompanying materials are made available under the
 * terms of the Eclipse Public License 2.0 which is available at
 * http://www.eclipse.org/legal/epl-2.0.
 * 
 * SPDX-License-Identifier: EPL-2.0
 */
package org.eclipse.elk.alg.rectpacking.options;

import java.util.EnumSet;
import org.eclipse.elk.alg.rectpacking.RectPackingLayoutProvider;
import org.eclipse.elk.alg.rectpacking.options.OptimizationGoal;
import org.eclipse.elk.alg.rectpacking.options.RectPackingMetaDataProvider;
import org.eclipse.elk.core.AbstractLayoutProvider;
import org.eclipse.elk.core.data.ILayoutMetaDataProvider;
import org.eclipse.elk.core.data.LayoutAlgorithmData;
import org.eclipse.elk.core.math.ElkPadding;
import org.eclipse.elk.core.options.ContentAlignment;
import org.eclipse.elk.core.options.CoreOptions;
import org.eclipse.elk.graph.properties.IProperty;
import org.eclipse.elk.graph.properties.Property;

/**
 * @rectpacking.md
 */
@SuppressWarnings("all")
public class RectPackingOptions implements ILayoutMetaDataProvider {
  /**
   * The id of the ELK Rectangle Packing algorithm.
   */
  public static final String ALGORITHM_ID = "org.eclipse.elk.rectpacking";
  
  /**
   * Default value for {@link #ASPECT_RATIO} with algorithm "ELK Rectangle Packing".
   */
  private static final double ASPECT_RATIO_DEFAULT = 1.3;
  
  /**
   * The desired aspect ratio of the drawing, that is the quotient of width by height.
   */
  public static final IProperty<Double> ASPECT_RATIO = new Property<Double>(
                                CoreOptions.ASPECT_RATIO,
                                ASPECT_RATIO_DEFAULT);
  
  /**
   * If active, nodes are expanded to fill the area of their parent.
   */
  public static final IProperty<Boolean> EXPAND_NODES = CoreOptions.EXPAND_NODES;
  
  /**
   * Default value for {@link #PADDING} with algorithm "ELK Rectangle Packing".
   */
  private static final ElkPadding PADDING_DEFAULT = new ElkPadding(15);
  
  /**
   * The padding to be left to a parent element's border when placing child elements. This can
   * also serve as an output option of a layout algorithm if node size calculation is setup
   * appropriately.
   */
  public static final IProperty<ElkPadding> PADDING = new Property<ElkPadding>(
                                CoreOptions.PADDING,
                                PADDING_DEFAULT);
  
  /**
   * Default value for {@link #SPACING_NODE_NODE} with algorithm "ELK Rectangle Packing".
   */
  private static final double SPACING_NODE_NODE_DEFAULT = 15;
  
  /**
   * The minimal distance to be preserved between each two nodes.
   */
  public static final IProperty<Double> SPACING_NODE_NODE = new Property<Double>(
                                CoreOptions.SPACING_NODE_NODE,
                                SPACING_NODE_NODE_DEFAULT);
  
  /**
   * Specifies how the content of compound nodes is to be aligned, e.g. top-left.
   */
  public static final IProperty<EnumSet<ContentAlignment>> CONTENT_ALIGNMENT = CoreOptions.CONTENT_ALIGNMENT;
  
  /**
   * Optimization goal for approximation of the bounding box given by the first iteration. Determines whether layout is
   * sorted by the maximum scaling, aspect ratio, or area. Depending on the strategy the aspect ratio might be nearly ignored.
   */
  public static final IProperty<OptimizationGoal> OPTIMIZATION_GOAL = RectPackingMetaDataProvider.OPTIMIZATION_GOAL;
  
  /**
   * When placing a rectangle behind or below the last placed rectangle in the first iteration, it is sometimes
   * possible to shift the rectangle further to the left or right, resulting in less whitespace. True (default)
   * enables the shift and false disables it. Disabling the shift produces a greater approximated area by the first
   * iteration and a layout, when using ONLY the first iteration (default not the case), where it is sometimes
   * impossible to implement a size transformation of rectangles that will fill the bounding box and eliminate
   * empty spaces.
   */
  public static final IProperty<Boolean> LAST_PLACE_SHIFT = RectPackingMetaDataProvider.LAST_PLACE_SHIFT;
  
  /**
   * If enabled only the width approximation step is executed and the nodes are placed accordingly.
   * The nodes are layouted according to the packingStrategy.
   * If set to true not expansion of nodes is taking place.
   */
  public static final IProperty<Boolean> ONLY_FIRST_ITERATION = RectPackingMetaDataProvider.ONLY_FIRST_ITERATION;
  
  /**
   * Enables compaction. Compacts blocks if they do not use the full height of the row.
   * This option allows to have a smaller drawing.
   * If this option is disabled all nodes are placed next to each other in rows.
   */
  public static final IProperty<Boolean> ROW_COMPACTION = RectPackingMetaDataProvider.ROW_COMPACTION;
  
  /**
   * Expands nodes if expandNodes is true to fit the aspect ratio instead of only in their bounds.
   * The option is only useful if the used packingStrategy is ASPECT_RATIO_DRIVEN, otherwise this may result
   * in unreasonable ndoe expansion.
   */
  public static final IProperty<Boolean> EXPAND_TO_ASPECT_RATIO = RectPackingMetaDataProvider.EXPAND_TO_ASPECT_RATIO;
  
  /**
   * Whether the algorithm should be run in interactive mode for the content of a parent node.
   * What this means exactly depends on how the specific algorithm interprets this option.
   * Usually in the interactive mode algorithms try to modify the current layout as little as
   * possible.
   */
  public static final IProperty<Boolean> INTERACTIVE = CoreOptions.INTERACTIVE;
  
  /**
   * Whether the graph should be changeable interactively and by setting constraints
   */
  public static final IProperty<Boolean> INTERACTIVE_LAYOUT = CoreOptions.INTERACTIVE_LAYOUT;
  
  /**
   * The rectangles are ordered. Normally according to their definition the the model.
   * This option allows to specify a desired position that has preference over the
   * original position.
   */
  public static final IProperty<Integer> DESIRED_POSITION = RectPackingMetaDataProvider.DESIRED_POSITION;
  
  /**
   * The rectangles are ordered. Normally according to their definition the the model.
   * This option specifies the current position of a node.
   */
  public static final IProperty<Integer> CURRENT_POSITION = RectPackingMetaDataProvider.CURRENT_POSITION;
  
  /**
   * Option to place the rectangles in the given target width
   * instead of approximating the width using the desired aspect ratio.
   */
  public static final IProperty<Double> TARGET_WIDTH = RectPackingMetaDataProvider.TARGET_WIDTH;
  
  /**
   * Layouter-specific algorithm factory.
   */
  public static class RectpackingFactory implements org.eclipse.elk.core.util.IFactory<AbstractLayoutProvider> {
    public AbstractLayoutProvider create() {
      AbstractLayoutProvider provider = new RectPackingLayoutProvider();
      provider.initialize("");
      return provider;
    }
    
    public void destroy(final AbstractLayoutProvider obj) {
      obj.dispose();
    }
  }
  
  public void apply(final org.eclipse.elk.core.data.ILayoutMetaDataProvider.Registry registry) {
    registry.register(new LayoutAlgorithmData.Builder()
        .id("org.eclipse.elk.rectpacking")
        .name("ELK Rectangle Packing")
        .description("Algorithm for packing of unconnected boxes, i.e. graphs without edges. The given order of the boxes is always preserved and the main reading direction of the boxes is left to right. The algorithm is divided into two phases. One phase approximates the width in which the rectangles can be placed. The next phase places the rectangles in rows using the previously calculated width as bounding width and bundles rectangles with a similar height in blocks. A compaction step reduces the size of the drawing. Finally, the rectangles are expanded to fill their bounding box and eliminate empty unused spaces.")
        .providerFactory(new RectpackingFactory())
        .melkBundleName(null)
        .definingBundleId("org.eclipse.elk.alg.rectpacking")
        .create()
    );
    registry.addOptionSupport(
        "org.eclipse.elk.rectpacking",
        "org.eclipse.elk.aspectRatio",
        ASPECT_RATIO_DEFAULT
    );
    registry.addOptionSupport(
        "org.eclipse.elk.rectpacking",
        "org.eclipse.elk.expandNodes",
        EXPAND_NODES.getDefault()
    );
    registry.addOptionSupport(
        "org.eclipse.elk.rectpacking",
        "org.eclipse.elk.padding",
        PADDING_DEFAULT
    );
    registry.addOptionSupport(
        "org.eclipse.elk.rectpacking",
        "org.eclipse.elk.spacing.nodeNode",
        SPACING_NODE_NODE_DEFAULT
    );
    registry.addOptionSupport(
        "org.eclipse.elk.rectpacking",
        "org.eclipse.elk.contentAlignment",
        CONTENT_ALIGNMENT.getDefault()
    );
    registry.addOptionSupport(
        "org.eclipse.elk.rectpacking",
        "org.eclipse.elk.rectpacking.optimizationGoal",
        OPTIMIZATION_GOAL.getDefault()
    );
    registry.addOptionSupport(
        "org.eclipse.elk.rectpacking",
        "org.eclipse.elk.rectpacking.lastPlaceShift",
        LAST_PLACE_SHIFT.getDefault()
    );
    registry.addOptionSupport(
        "org.eclipse.elk.rectpacking",
        "org.eclipse.elk.rectpacking.onlyFirstIteration",
        ONLY_FIRST_ITERATION.getDefault()
    );
    registry.addOptionSupport(
        "org.eclipse.elk.rectpacking",
        "org.eclipse.elk.rectpacking.rowCompaction",
        ROW_COMPACTION.getDefault()
    );
    registry.addOptionSupport(
        "org.eclipse.elk.rectpacking",
        "org.eclipse.elk.rectpacking.expandToAspectRatio",
        EXPAND_TO_ASPECT_RATIO.getDefault()
    );
    registry.addOptionSupport(
        "org.eclipse.elk.rectpacking",
        "org.eclipse.elk.interactive",
        INTERACTIVE.getDefault()
    );
    registry.addOptionSupport(
        "org.eclipse.elk.rectpacking",
        "org.eclipse.elk.interactiveLayout",
        INTERACTIVE_LAYOUT.getDefault()
    );
    registry.addOptionSupport(
        "org.eclipse.elk.rectpacking",
        "org.eclipse.elk.rectpacking.desiredPosition",
        DESIRED_POSITION.getDefault()
    );
    registry.addOptionSupport(
        "org.eclipse.elk.rectpacking",
        "org.eclipse.elk.rectpacking.currentPosition",
        CURRENT_POSITION.getDefault()
    );
    registry.addOptionSupport(
        "org.eclipse.elk.rectpacking",
        "org.eclipse.elk.rectpacking.targetWidth",
        TARGET_WIDTH.getDefault()
    );
  }
}
