/**
 * Copyright (c) 2018 Kiel University and others.
 * 
 * This program and the accompanying materials are made available under the
 * terms of the Eclipse Public License 2.0 which is available at
 * http://www.eclipse.org/legal/epl-2.0.
 * 
 * SPDX-License-Identifier: EPL-2.0
 */
package org.eclipse.elk.alg.packing.rectangles.options;

import org.eclipse.elk.alg.packing.rectangles.RectPackingLayoutProvider;
import org.eclipse.elk.alg.packing.rectangles.options.RectPackingMetaDataProvider;
import org.eclipse.elk.alg.packing.rectangles.util.PackingStrategy;
import org.eclipse.elk.core.AbstractLayoutProvider;
import org.eclipse.elk.core.data.ILayoutMetaDataProvider;
import org.eclipse.elk.core.data.LayoutAlgorithmData;
import org.eclipse.elk.core.math.ElkPadding;
import org.eclipse.elk.core.options.CoreOptions;
import org.eclipse.elk.graph.properties.IProperty;
import org.eclipse.elk.graph.properties.Property;

@SuppressWarnings("all")
public class RectPackingOptions implements ILayoutMetaDataProvider {
  /**
   * The id of the Rectangle Layout. algorithm.
   */
  public static final String ALGORITHM_ID = "org.eclipse.elk.rectpacking.rectPacking";
  
  /**
   * Default value for {@link #ASPECT_RATIO} with algorithm "Rectangle Layout.".
   */
  private static final double ASPECT_RATIO_DEFAULT = 1.3;
  
  /**
   * The desired aspect ratio of the drawing, that is the quotient of width by height.
   */
  public static final IProperty<Double> ASPECT_RATIO = new Property<Double>(
                                CoreOptions.ASPECT_RATIO,
                                ASPECT_RATIO_DEFAULT);
  
  /**
   * Default value for {@link #PADDING} with algorithm "Rectangle Layout.".
   */
  private static final ElkPadding PADDING_DEFAULT = new ElkPadding(15);
  
  /**
   * The padding to be left to a parent element's border when placing child elements. This can
   * also serve as an output option of a layout algorithm if node size calculation is setup
   * appropriately.
   */
  public static final IProperty<ElkPadding> PADDING = new Property<ElkPadding>(
                                CoreOptions.PADDING,
                                PADDING_DEFAULT);
  
  /**
   * If active, nodes are expanded to fill the area of their parent.
   */
  public static final IProperty<Boolean> EXPAND_NODES = CoreOptions.EXPAND_NODES;
  
  /**
   * Strategy for approximation of the bounding box given by the first iteration. Determines whether layout is
   * sorted by scale measure, aspect ratio or area.
   */
  public static final IProperty<PackingStrategy> STRATEGY = RectPackingMetaDataProvider.STRATEGY;
  
  /**
   * When placing a rectangle behind or below the last placed rectangle in the first iteration, it is sometimes
   * possible to shift the rectangle further to the left or right, resulting in less whitespace. True (default)
   * enables the shift and false disables it. Disabling the shift produces a greater approximated area by the first
   * iteration and a layout, when using ONLY the first iteration (default not the case), where it is sometimes
   * impossible to implement a size transformation of rectangles that will fill the bounding box and eliminate
   * empty spaces.
   */
  public static final IProperty<Boolean> LAST_PLACE_SHIFT = RectPackingMetaDataProvider.LAST_PLACE_SHIFT;
  
  /**
   * Determines whether only the first iteration should be executed or not. If set to true, only the first iteration
   * will be executed. When using only the first iteration, rectangles will not be resized to fill their bounding
   * box and eliminate unused whitespace.
   */
  public static final IProperty<Boolean> ONLY_FIRST_ITERATION = RectPackingMetaDataProvider.ONLY_FIRST_ITERATION;
  
  /**
   * Determines whether the algorithm should check for the case that one big rectangle is present with a lot of
   * smaller rectangles of the same size. If the check is true, a special algorithm for this case is used.
   */
  public static final IProperty<Boolean> CHECK_FOR_SPECIAL_CASE = RectPackingMetaDataProvider.CHECK_FOR_SPECIAL_CASE;
  
  /**
   * Layouter-specific algorithm factory.
   */
  public static class RectPackingFactory implements org.eclipse.elk.core.util.IFactory<AbstractLayoutProvider> {
    public AbstractLayoutProvider create() {
      AbstractLayoutProvider provider = new RectPackingLayoutProvider();
      provider.initialize("");
      return provider;
    }
    
    public void destroy(final AbstractLayoutProvider obj) {
      obj.dispose();
    }
  }
  
  public void apply(final org.eclipse.elk.core.data.ILayoutMetaDataProvider.Registry registry) {
    registry.register(new LayoutAlgorithmData.Builder()
        .id("org.eclipse.elk.rectpacking.rectPacking")
        .name("Rectangle Layout.")
        .description("Algorithm for packing of unconnected boxes, i.e. graphs without edges. In the default settings, the algorithm checks for a case where one big rectangle is present with a lot of smaller rectangles with the same height and applies a special algorithm to it. If this is not the case, the algorithm then uses to phases. One phase approximates an area where the rectangles can be placed and then another to place them there and align them nicely. Finally, the rectangles are expanded to fill their bounding box and eliminate empty unused spaces.")
        .providerFactory(new RectPackingFactory())
        .melkBundleName(null)
        .definingBundleId("org.eclipse.elk.alg.packing.rectangles")
        .create()
    );
    registry.addOptionSupport(
        "org.eclipse.elk.rectpacking.rectPacking",
        "org.eclipse.elk.aspectRatio",
        ASPECT_RATIO_DEFAULT
    );
    registry.addOptionSupport(
        "org.eclipse.elk.rectpacking.rectPacking",
        "org.eclipse.elk.padding",
        PADDING_DEFAULT
    );
    registry.addOptionSupport(
        "org.eclipse.elk.rectpacking.rectPacking",
        "org.eclipse.elk.expandNodes",
        EXPAND_NODES.getDefault()
    );
    registry.addOptionSupport(
        "org.eclipse.elk.rectpacking.rectPacking",
        "org.eclipse.elk.rectpacking.strategy",
        STRATEGY.getDefault()
    );
    registry.addOptionSupport(
        "org.eclipse.elk.rectpacking.rectPacking",
        "org.eclipse.elk.rectpacking.lastPlaceShift",
        LAST_PLACE_SHIFT.getDefault()
    );
    registry.addOptionSupport(
        "org.eclipse.elk.rectpacking.rectPacking",
        "org.eclipse.elk.rectpacking.onlyFirstIteration",
        ONLY_FIRST_ITERATION.getDefault()
    );
    registry.addOptionSupport(
        "org.eclipse.elk.rectpacking.rectPacking",
        "org.eclipse.elk.rectpacking.checkForSpecialCase",
        CHECK_FOR_SPECIAL_CASE.getDefault()
    );
  }
}
